package ru.yandex.market.graphouse.server;

import java.util.List;
import java.util.Queue;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.Semaphore;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.market.graphouse.cacher.MetricCacher;
import ru.yandex.market.graphouse.cacher.MetricCacherProvider;
import ru.yandex.monlib.metrics.primitives.GaugeInt64;
import ru.yandex.monlib.metrics.registry.MetricRegistry;

/**
 * Here live metrics with ids not yet synced to db
 * @author Maksim Leonov (nohttp@)
 */
@Component
@ParametersAreNonnullByDefault
public class MetricLimbo {

    private static final Logger log = LoggerFactory.getLogger(MetricLimbo.class);
    private static final int LIMBO_SIZE = 4_000_000;

    private final Semaphore freeSlots = new Semaphore(LIMBO_SIZE);
    private final Queue<List<UnresolvedMetric>> unresolvedMetrics = new ConcurrentLinkedQueue<>();
    private final MetricCacher cacher;
    private final GaugeInt64 notResolvedMetricsGauge;
    private final GaugeInt64 resolvedMetricsGauge;

    @Autowired
    public MetricLimbo(MetricCacherProvider metricCacherProvider) {
        this.cacher = metricCacherProvider.makeCacher("Limbo-0");
        MetricRegistry registry = MetricRegistry.root();
        this.notResolvedMetricsGauge = registry.gaugeInt64("MetricLimbo.unresolvedMetrics");
        this.resolvedMetricsGauge = registry.gaugeInt64("MetricLimbo.resolvedMetrics");
        registry.lazyGaugeInt64("MetricLimbo.queueSize", () -> LIMBO_SIZE - freeSlots.availablePermits());
    }

    void submitMetrics(List<UnresolvedMetric> metrics) {
        if (metrics.isEmpty()) {
            return;
        }
        try {
            freeSlots.acquire(metrics.size());
            unresolvedMetrics.add(metrics);
        } catch (InterruptedException e) {
            throw new RuntimeException(e);
        }
    }

    public void onDbUpdateFetched() {
        List<UnresolvedMetric> stillNotResolvedMetrics = UnresolvedMetric.f.newInstance();
        List<UnresolvedMetric> unresolvedBatch;

        MetricBatch resolvedBatch = new MetricBatch(LIMBO_SIZE - freeSlots.availablePermits());

        while ((unresolvedBatch = unresolvedMetrics.poll()) != null) {
            for (UnresolvedMetric unresolvedMetric : unresolvedBatch) {
                if (unresolvedMetric.isSyncedWithDb()) {
                    resolvedBatch.add(unresolvedMetric.asResolved());
                } else {
                    stillNotResolvedMetrics.add(unresolvedMetric);
                }
            }
        }
        log.info("Limbo iteration: " + resolvedBatch.size() + " points resolved, " + stillNotResolvedMetrics.size() + " not resolved yet.");
        notResolvedMetricsGauge.set(stillNotResolvedMetrics.size());
        resolvedMetricsGauge.set(resolvedBatch.size());
        if (!stillNotResolvedMetrics.isEmpty()) {
            unresolvedMetrics.add(stillNotResolvedMetrics);
        }
        if (!resolvedBatch.isEmpty()) {
            cacher.submitMetrics(resolvedBatch);
            freeSlots.release(resolvedBatch.size());
        }
    }
}
