package ru.yandex.market.graphouse.server;

import java.io.IOException;
import java.util.regex.Pattern;

import javax.annotation.Nullable;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import ru.yandex.market.graphouse.PatternListFile;
import ru.yandex.market.graphouse.search.MetricResponseStatus;
import ru.yandex.monlib.metrics.primitives.Rate;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.staffOnly.annotations.LinkedOnRootPage;
import ru.yandex.solomon.staffOnly.annotations.ManagerMethod;

/**
 * @author Dmitry Andreev <a href="mailto:AndreevDm@yandex-team.ru"/>
 * @date 27/05/15
 */
@Component
@LinkedOnRootPage("Switch Metric Validation")
public class MetricValidator {

    private static final Logger logger = LoggerFactory.getLogger(MetricValidator.class);

    private static int minMetricLength = 10;
    private static int maxMetricLength = 255;
    private static int minDots = 2;
    private static int maxDots = 15;
    private static final String metricRegexp = "^[-_0-9a-zA-Z\\.:=/\\[\\]()\"?]+$";
    private static final Pattern metricPattern = Pattern.compile(metricRegexp);
    // TODO: move to kikimr @alexlovkov
    private static final String BLACKLIST_FILE_NAME = "bannedMetrics.txt";
    public static final MetricValidator DEFAULT = new MetricValidator();

    private final Rate blackListRate;
    private PatternListFile blacklist;

    public MetricValidator() {
        this.blacklist = new PatternListFile(BLACKLIST_FILE_NAME);
        MetricRegistry metricRegistry = MetricRegistry.root();
        this.blackListRate = metricRegistry.rate("metricValidator.blackList");
    }

    public MetricResponseStatus validate(String name, String ipAddress) {
        return validate(name, false, ipAddress);
    }

    public MetricResponseStatus validate(String metric, boolean allowDirs, @Nullable String ipAddress) {
        if (metric.length() < minMetricLength || metric.length() > maxMetricLength) {
            return MetricResponseStatus.REJECTED_LENGTH;
        }
        if (!validateDots(metric, allowDirs)) {
            return MetricResponseStatus.REJECTED_DOUBLE_DOTS;
        }
        if (!metricPattern.matcher(metric).matches()) {
            return MetricResponseStatus.REJECTED_PATTERN;
        }
        if (blacklist.isMatch(metric, ipAddress)) {
            blackListRate.inc();
            return MetricResponseStatus.BLACKLIST_METRIC;
        }
        return MetricResponseStatus.OK;
    }

    private boolean validateDots(String name, boolean allowDirs) {
        if (name.charAt(0) == '.') {
            return false;
        }
        if (!allowDirs && name.charAt(name.length() - 1) == '.') {
            return false;
        }
        int prevDotIndex = -1;
        int dotIndex = -1;
        int dotCount = 0;
        while ((dotIndex = name.indexOf('.', prevDotIndex + 1)) > 0) {
            if (prevDotIndex + 1 == dotIndex) {
                return false; //Две точки подряд
            }
            prevDotIndex = dotIndex;
            dotCount++;
        }
        if (dotCount < minDots || dotCount > maxDots) {
            return false;
        }
        return true;
    }

    @ManagerMethod
    public void addBlacklistPattern(String pattern) {
        blacklist.addPattern(pattern);
    }

    @ManagerMethod
    public void removeBlacklistPattern(String pattern) {
        blacklist.removePattern(pattern);
    }

    @ManagerMethod
    public void updateBlacklistPattern(String oldPattern, String newPattern) {
        blacklist.updatePattern(oldPattern, newPattern);
    }

    /**
     * for test purpose only
     */
    void clearFile() throws IOException {
        blacklist.clearFile();
    }

}
