package ru.yandex.market.graphouse.stockpile;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import it.unimi.dsi.fastutil.ints.Int2ObjectMap;
import it.unimi.dsi.fastutil.ints.Int2ObjectOpenHashMap;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.market.graphouse.search.MetricResponseStatus;
import ru.yandex.market.graphouse.search.tree.MetricNameZip;
import ru.yandex.market.graphouse.server.MetricBatch;
import ru.yandex.market.graphouse.server.ResolvedMetric;
import ru.yandex.market.graphouse.stockpile.proxy.GraphiteStockpileClient;
import ru.yandex.market.graphouse.stockpile.proxy.ReadRequest;
import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.primitives.Rate;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.config.protobuf.graphite.storage.TGraphiteStorageConfig;
import ru.yandex.solomon.model.point.AggrPoint;
import ru.yandex.solomon.model.point.column.ValueColumn;
import ru.yandex.solomon.model.protobuf.MetricType;
import ru.yandex.solomon.model.timeseries.AggrGraphDataArrayList;
import ru.yandex.solomon.selfmon.AvailabilityStatus;
import ru.yandex.stockpile.api.EProjectId;
import ru.yandex.stockpile.client.shard.StockpileMetricId;
import ru.yandex.stockpile.client.writeRequest.StockpileShardWriteRequest;
import ru.yandex.stockpile.client.writeRequest.StockpileShardWriteRequestBuilder;

import static java.util.stream.Collectors.toList;

/**
 * @author Maksim Leonov (nohttp@)
 */
@Component
public class GraphouseStockpileClient implements GraphouseStockpileIdGenerator {

    private static final int MAX_BATCH_WRITE_SIZE_IN_BYTES = 50 << 20; // 50 MiB

    private static final Logger log = LoggerFactory.getLogger(GraphouseStockpileClient.class);

    private final GraphiteStockpileClient stockpileClient;
    private final int stockpileOwnerShardId;

    private final Rate sentToStockpileRate;
    private final Rate storedInStockpileRate;
    private final Rate failedInStockpileRate;

    @Autowired
    public GraphouseStockpileClient(
        GraphiteStockpileClient stockpileClient,
        TGraphiteStorageConfig config)
    {
        this.stockpileClient = stockpileClient;
        this.stockpileOwnerShardId = config.getStockpileOwnerShardId();

        MetricRegistry metricRegistry = MetricRegistry.root();

        sentToStockpileRate = metricRegistry.rate("writeStatus",
            Labels.of("status", MetricResponseStatus.IN_FLIGHT_STOCKPILE.name()));
        storedInStockpileRate = metricRegistry.rate("writeStatus",
            Labels.of("status", MetricResponseStatus.OK.name()));
        failedInStockpileRate = metricRegistry.rate("writeStatus",
            Labels.of("status", MetricResponseStatus.ERROR_STOCKPILE.name()));
    }

    public CompletableFuture<Void> saveMetrics(StockpilePushBatch metrics) {
        if (!stockpileClient.isFullyReady()) {
            throw new RuntimeException("Stockpile client is not ready");
        }

        var writes = metrics.getWritesByShard();
        var futures = new ArrayList<CompletableFuture<Void>>(writes.size());

        for (var it = writes.int2ObjectEntrySet().fastIterator(); it.hasNext(); ) {
            var entry = it.next();

            StockpileShardWriteRequest request = entry.getValue();
            if (request.getData() == null) {
                // do not send request to Stockpile if it was successfully written on previous retry
                it.remove();
                continue;
            }

            var future = stockpileClient.writeData(entry.getIntKey(), request)
                    .whenComplete((aVoid, throwable) -> {
                        if (throwable == null) {
                            request.close();
                        }
                    });
            futures.add(future);
        }

        sentToStockpileRate.add(metrics.size());
        return CompletableFutures.allOfVoid(futures).whenComplete((unit, throwable) -> {
            if (throwable != null) {
                failedInStockpileRate.add(metrics.size());
            } else {
                storedInStockpileRate.add(metrics.size());
            }
        });
    }

    private int addMetricTo(ResolvedMetric metric, Int2ObjectMap<StockpileShardWriteRequestBuilder> writesByShard) {
        int shardId = metric.stockpileShardId;
        var builder = writesByShard.get(shardId);
        if (builder == null) {
            builder = new StockpileShardWriteRequestBuilder(EProjectId.GRAPHITE, stockpileOwnerShardId);
            writesByShard.put(shardId, builder);
        }

        AggrPoint point = new AggrPoint();
        point.setValue(metric.value, ValueColumn.DEFAULT_DENOM);
        point.setTsMillis(metric.tsMillis);

        int oldSize = builder.bytesSize();
        builder.addRecord(
            metric.stockpileLocalId,
            point,
            metric.stockpileDecimPolicyId,
            MetricType.DGAUGE
        );
        return builder.bytesSize() - oldSize;
    }

    @Override
    public StockpileMetricId generateMetricId() {
        var result = stockpileClient.generateMetricId();
        log.info("New stockpile ID requested; generated: {}", result);
        return result;
    }

    @Override
    public boolean isStockpileIdFromDatabase() {
        return false;
    }

    public CompletableFuture<List<MetricResponse>> readMetrics(MetricNameZip[] requests, long fromMillis, long toMillis) {
        return Stream.of(requests)
            .map(metric -> readOne(metric, fromMillis, toMillis))
            .collect(Collectors.collectingAndThen(toList(), CompletableFutures::allOf))
            .thenApply(listF -> listF);
    }

    private CompletableFuture<MetricResponse> readOne(MetricNameZip metric, long fromMillis, long toMillis) {
        return stockpileClient.readOne(ReadRequest.newBuilder()
            .setKey(metric.getStockpileId())
            .setFromMillis(fromMillis)
            .setToMillis(toMillis)
            .build())
            .thenApply(response -> {
               if (!response.isOk()) {
                   throw new RuntimeException(metric.getName() + " " + response.errorMessage());
               }

               var payload = AggrGraphDataArrayList.of(response.getSource()).toGraphDataShort(response.getDataType());
               return new MetricResponse(metric, payload);
            });
    }

    public AvailabilityStatus getAvailability() {
        return stockpileClient.getStatus();
    }

    public List<StockpilePushBatch> compress(ArrayList<MetricBatch> buffer) {
        List<StockpilePushBatch> batches = new ArrayList<>();
        var writesByShard = new Int2ObjectOpenHashMap<StockpileShardWriteRequestBuilder>();

        int metricsCount = 0;
        long size = 0;
        for (MetricBatch batch : buffer) {
            for (ResolvedMetric resolvedMetric : batch.getMetrics()) {
                size += addMetricTo(resolvedMetric, writesByShard);
                metricsCount++;
                if (size > MAX_BATCH_WRITE_SIZE_IN_BYTES) {
                    batches.add(new StockpilePushBatch(writesByShard, metricsCount));
                    writesByShard = new Int2ObjectOpenHashMap<>(writesByShard.size());
                    metricsCount = 0;
                    size = 0;
                }
            }
        }
        batches.add(new StockpilePushBatch(writesByShard, metricsCount));
        return batches;
    }
}
