package ru.yandex.market.graphouse.stockpile;

import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

import com.google.common.collect.Iterables;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;

import ru.yandex.discovery.cluster.ClusterMapper;
import ru.yandex.discovery.cluster.ClusterMapperContext;
import ru.yandex.grpc.conf.ClientOptionsFactory;
import ru.yandex.grpc.conf.GrpcConfigurationContext;
import ru.yandex.market.graphouse.retention.HardcodedRetentionManager;
import ru.yandex.market.graphouse.stockpile.proxy.CrossDcStockpileClient;
import ru.yandex.market.graphouse.stockpile.proxy.DcStockpileClient;
import ru.yandex.market.graphouse.stockpile.proxy.GraphiteStockpileClient;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.config.protobuf.graphite.storage.TGraphiteStorageConfig;
import ru.yandex.solomon.config.thread.ThreadPoolProvider;
import ru.yandex.stockpile.client.StockpileClientFactory;

/**
 * @author Maksim Leonov (nohttp@)
 */
@Configuration
@Import({
    GraphouseStockpileClient.class,
    HardcodedRetentionManager.class,
    ClusterMapperContext.class,
    GrpcConfigurationContext.class
})
public class GraphouseStockpileContext {

    private final TGraphiteStorageConfig config;
    private final ClusterMapper clusterMapper;

    @Autowired
    public GraphouseStockpileContext(TGraphiteStorageConfig config, ClusterMapper clusterMapper) {
        this.config = config;
        this.clusterMapper = clusterMapper;
    }

    @Bean
    public StockpileClientFactory stockpileClientFactory(
            ThreadPoolProvider threads,
            ClientOptionsFactory clientOptionsFactory)
    {
        return new StockpileClientFactory(threads, clusterMapper, MetricRegistry.root(), clientOptionsFactory);
    }

    @Bean
    public GraphiteStockpileClient stockpileClient(StockpileClientFactory factory) {

        Map<String, DcStockpileClient> clients = factory.createClients(config.getClientId(), config.getStockpileClientConfig())
            .entrySet()
            .stream()
            .map(entry -> Map.entry(entry.getKey(), new DcStockpileClient(entry.getValue())))
            .collect(Collectors.toMap(Map.Entry::getKey, Map.Entry::getValue));

        if (clients.isEmpty()) {
            throw new IllegalStateException("No configured stockpile client");
        }

        if (clients.size() == 1) {
            return Iterables.getOnlyElement(clients.values());
        }

        if (clients.size() > 2) {
            throw new IllegalStateException("Too many stockpile dc(StockpileClientConfig): " + clients.size());
        }

        var masterId = config.getStockpileMasterClusterId();
        DcStockpileClient master = Objects.requireNonNull(clients.get(masterId),
            "StockpileMasterClusterId not found stockpile client");
        var slave = clients.entrySet()
            .stream()
            .filter(entry -> !entry.getKey().equals(masterId))
            .map(Map.Entry::getValue)
            .findFirst();

        if (slave.isEmpty()) {
            return master;
        }

        return new CrossDcStockpileClient(master, slave.get());
    }
}
