package ru.yandex.market.graphouse.stockpile.proxy;

import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ThreadLocalRandom;
import java.util.stream.Collectors;

import com.google.common.base.Throwables;
import com.google.common.primitives.Longs;
import com.google.protobuf.UnsafeByteOperations;

import ru.yandex.solomon.codec.archive.serializer.MetricArchiveNakedSerializer;
import ru.yandex.solomon.codec.serializer.StockpileDeserializer;
import ru.yandex.solomon.codec.serializer.StockpileFormat;
import ru.yandex.solomon.model.protobuf.MetricId;
import ru.yandex.solomon.model.protobuf.TimeSeries;
import ru.yandex.solomon.model.timeseries.AggrGraphDataIterable;
import ru.yandex.solomon.model.timeseries.ConcatAggrGraphDataIterable;
import ru.yandex.solomon.selfmon.AvailabilityStatus;
import ru.yandex.stockpile.api.EStockpileStatusCode;
import ru.yandex.stockpile.api.MetricMeta;
import ru.yandex.stockpile.api.ReadMetricsMetaRequest;
import ru.yandex.stockpile.api.TReadRequest;
import ru.yandex.stockpile.api.TWriteDataBinaryRequest;
import ru.yandex.stockpile.client.StockpileClient;
import ru.yandex.stockpile.client.shard.StockpileLocalId;
import ru.yandex.stockpile.client.shard.StockpileMetricId;
import ru.yandex.stockpile.client.writeRequest.StockpileShardWriteRequest;

import static java.util.stream.Collectors.toList;

/**
 * @author Vladimir Gordiychuk
 */
public class DcStockpileClient implements GraphiteStockpileClient {

    private final StockpileClient client;

    public DcStockpileClient(StockpileClient client) {
        this.client = client;
    }

    @Override
    public StockpileMetricId generateMetricId() {
        int shardId = ThreadLocalRandom.current().nextInt(1, client.getTotalShardsCount() + 1);
        long localId = StockpileLocalId.random();
        return new StockpileMetricId(shardId, localId);
    }

    @Override
    public int getTotalShardsCount() {
        return client.getTotalShardsCount();
    }

    @Override
    public boolean isFullyReady() {
        return client.getReadyShardsCount() > 0;
    }

    @Override
    public CompletableFuture<ReadResponse> readOne(ReadRequest request) {
        var format = defineStockpileFormat();
        var key = request.getKey();
        return client.readCompressedOne(TReadRequest.newBuilder()
            .setMetricId(MetricId.newBuilder()
                .setShardId(key.getShardId())
                .setLocalId(key.getLocalId())
                .build())
            .setFromMillis(request.getFromMillis())
            .setToMillis(request.getToMillis())
            .setBinaryVersion(format.getFormat())
            .setDeadline(request.getDeadline())
            .build())
            .thenApply(response -> {
                if (response.getStatus() != EStockpileStatusCode.OK) {
                    return new ReadResponse(key, response.getStatus(), response.getStatusMessage());
                }

                var target = decodeChunks(format, response.getChunksList());
                return new ReadResponse(key, response.getType(), target);
            })
            .exceptionally(e -> new ReadResponse(key, EStockpileStatusCode.INTERNAL_ERROR, Throwables.getStackTraceAsString(e)));
    }

    @Override
    public AvailabilityStatus getStatus() {
        return client.getAvailability();
    }

    @Override
    public CompletableFuture<Void> writeData(int shardId, StockpileShardWriteRequest logEntry) {
        return client.writeDataBinary(TWriteDataBinaryRequest.newBuilder()
            .setShardId(shardId)
            .setContent(UnsafeByteOperations.unsafeWrap(logEntry.serialize()))
            .build())
            .thenApply(response -> {
                if (response.getStatus() != EStockpileStatusCode.OK) {
                    throw new RuntimeException(response.getStatus() + ": "+ response.getStatusMessage());
                }

                return null;
            });
    }

    @Override
    public CompletableFuture<List<MetricMeta>> readMetricsMeta(int shardId, long[] localIds) {
        return client.readMetricsMeta(ReadMetricsMetaRequest.newBuilder()
            .setShardId(shardId)
            .addAllLocalIds(Longs.asList(localIds))
            .build())
            .thenApply(response -> {
                if (response.getStatus() != EStockpileStatusCode.OK) {
                    throw new RuntimeException(response.getStatus() + ": " + response.getStatusMessage());
                }
                return response.getMetaList();
            });
    }

    private StockpileFormat defineStockpileFormat() {
        return StockpileFormat.byNumberOrCurrent(client.getCompatibleCompressFormat().upperEndpoint());
    }

    private AggrGraphDataIterable decodeChunks(StockpileFormat format, List<TimeSeries.Chunk> chunks) {
        return chunks.stream()
            .map(chunk -> decodeChunk(format, chunk))
            .collect(Collectors.collectingAndThen(toList(), ConcatAggrGraphDataIterable::of));
    }

    private AggrGraphDataIterable decodeChunk(StockpileFormat format, TimeSeries.Chunk chunk) {
        StockpileDeserializer deserializer = new StockpileDeserializer(chunk.getContent());
        return MetricArchiveNakedSerializer.serializerForFormatSealed(format).deserializeToEof(deserializer);
    }

    @Override
    public String toString() {
        return client.toString();
    }
}
