package ru.yandex.market.graphouse.stockpile.proxy;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.ThreadLocalRandom;

import ru.yandex.solomon.codec.archive.MetricArchiveMutable;
import ru.yandex.solomon.selfmon.AvailabilityStatus;
import ru.yandex.stockpile.api.MetricMeta;
import ru.yandex.stockpile.client.shard.StockpileLocalId;
import ru.yandex.stockpile.client.shard.StockpileMetricId;
import ru.yandex.stockpile.client.writeRequest.StockpileShardWriteRequest;

/**
 * @author alexlovkov
 */
public class GraphiteStockpileClientStub implements GraphiteStockpileClient {
    private final ConcurrentMap<StockpileMetricId, MetricArchiveMutable> archiveByKey = new ConcurrentHashMap<>();

    @Override
    public CompletableFuture<Void> writeData(int shardId, StockpileShardWriteRequest request) {
        var update = request.getDataByLocalId();
        for (var entry : update.entrySet()) {
            var id = new StockpileMetricId(shardId, entry.getKey());
            var archive = archiveByKey.computeIfAbsent(id, stockpileMetricId -> new MetricArchiveMutable());
            archive.updateWith(entry.getValue());
        }

        return CompletableFuture.completedFuture(null);
    }

    @Override
    public CompletableFuture<List<MetricMeta>> readMetricsMeta(int shardId, long[] localIds) {
        return CompletableFuture.supplyAsync(() -> {
            List<MetricMeta> result = new ArrayList<>(localIds.length);
            MetricMeta.Builder builder = MetricMeta.newBuilder(); // reuse builder
            for (long localId : localIds) {
                var id = new StockpileMetricId(shardId, localId);
                long lastTsMillis = archiveByKey.getOrDefault(id, new MetricArchiveMutable()).getLastTsMillis();
                result.add(builder
                    .setLocalId(localId)
                    .setLastTsMillis(lastTsMillis)
                    .build());
            }
            return result;
        });
    }

    @Override
    public StockpileMetricId generateMetricId() {
        int shardId = ThreadLocalRandom.current().nextInt(1, 5000);
        long localId = StockpileLocalId.random();
        return new StockpileMetricId(shardId, localId);
    }

    @Override
    public int getTotalShardsCount() {
        return 5000;
    }

    @Override
    public boolean isFullyReady() {
        return true;
    }

    @Override
    public CompletableFuture<ReadResponse> readOne(ReadRequest request) {
        return CompletableFuture.supplyAsync(() -> {
            var archive = archiveByKey.getOrDefault(request.getKey(), new MetricArchiveMutable());
            return new ReadResponse(request.getKey(), archive.getType(), archive);
        });
    }

    @Override
    public AvailabilityStatus getStatus() {
        return AvailabilityStatus.AVAILABLE;
    }
}
