package ru.yandex.market.graphouse;

import java.io.IOException;
import java.util.Map;
import java.util.concurrent.ArrayBlockingQueue;

import org.junit.After;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;

/**
 * @author alexlovkov
 */
public class PatternListFileTest {

    private static final String ipAddress = "127.0.0.1";
    private PatternListFile list;

    @Before
    public void createFile() {
        list = new PatternListFile("test.txt");
    }

    @After
    public void deleteFile() throws IOException {
        list.clearFile();
    }

    @Test
    public void inside() {
        list.addPattern("mail\\..*");
        Assert.assertTrue(list.isMatch("mail.", ipAddress));
        Assert.assertTrue(list.isMatch("mail.q", ipAddress));
        Assert.assertTrue(list.isMatch("mail..", ipAddress));
        Assert.assertTrue(list.isMatch("mail.mail", ipAddress));
        Assert.assertTrue(list.isMatch("mail.qweqweqwe", ipAddress));

        Assert.assertFalse(list.isMatch("mail", ipAddress));
        Assert.assertFalse(list.isMatch("mailq", ipAddress));
        Assert.assertFalse(list.isMatch("maill", ipAddress));
        Assert.assertFalse(list.isMatch(".mail.", ipAddress));
    }

    @Test
    public void testRemovePattern() {
        String deletedPattern = "aaa\\.bbb\\.ccc\\.ddd.*";
        list.addPattern(deletedPattern);
        list.removePattern(deletedPattern);
        list.addPattern("aaa\\.bbb\\.ccc\\.ddd\\.eee\\..*");
        Assert.assertFalse(list.isMatch("aaa.bbb.ccc.ddd", ipAddress));
        Assert.assertFalse(list.isMatch("aaa.bbb.ccc.ddd.qwe", ipAddress));
        Assert.assertTrue(list.isMatch("aaa.bbb.ccc.ddd.eee.e", ipAddress));
    }


    @Test
    public void testUpdatePattern() throws IOException {
        String updatedPattern = "aaa\\.bbb\\.ccc\\.ddd.*";
        list.addPattern(updatedPattern);
        list.updatePattern(updatedPattern, "aaa\\.bbb\\.ccc\\.gggg.*");
        list.addPattern("aaa\\.bbb\\.ccc\\.ddd\\.eee\\..*");
        Assert.assertFalse(list.isMatch("aaa.bbb.ccc.ddd.ee1", ipAddress));
        Assert.assertFalse(list.isMatch("aaa.bbb.ccc.ddd.qwe", ipAddress));

        Assert.assertTrue(list.isMatch("aaa.bbb.ccc.ddd.eee.e", ipAddress));
        Assert.assertTrue(list.isMatch("aaa.bbb.ccc.gggg.q1", ipAddress));
        Assert.assertTrue(list.isMatch("aaa.bbb.ccc.gggg.q1.q2", ipAddress));
    }

    @Test
    public void getStatistic() {
        list = new PatternListFile("test.txt", 3);
        String pattern = "aaa\\.bbb\\.ccc\\.ddd.*";
        list.addPattern(pattern);
        list.isMatch("aaa.bbb.ccc.ddd.e1", "127.0.0.1");
        list.isMatch("aaa.bbb.ccc.ddd.e1", "127.0.0.2");
        list.isMatch("aaa.bbb.ccc.ddd.e1", "127.0.0.3");

        Map<String, PatternListFile.StatisticMetric> statistic = list.getStatistic();
        ArrayBlockingQueue<String> queue = statistic.get(pattern).getLastIpAddresses();
        Assert.assertEquals(3, queue.size());
        validateIps(queue, "127.0.0.1", "127.0.0.2", "127.0.0.3");
        Assert.assertEquals(3, statistic.get(pattern).getCount());

        list.isMatch("aaa.bbb.ccc.ddd.e2", "127.0.0.3");
        list.isMatch("aaa.bbb.ccc.ddd.e3", "127.0.0.2");
        list.isMatch("aaa.bbb.ccc.ddd.e3", "127.0.0.1");
        list.isMatch("aaa.bbb.ccc.ddd.e3", "127.0.0.5");
        statistic = list.getStatistic();
        queue = statistic.get(pattern).getLastIpAddresses();
        Assert.assertEquals(3, queue.size());
        validateIps(queue, "127.0.0.2", "127.0.0.1", "127.0.0.5");
        Assert.assertEquals(7, statistic.get(pattern).getCount());
    }

    private void validateIps(ArrayBlockingQueue<String> queue, String... ips) {
        int index = 0;
        for (String s : queue) {
            Assert.assertEquals(s, ips[index]);
            index++;
        }
    }
}
