package ru.yandex.market.graphouse.search;

import java.io.IOException;
import java.nio.file.PathMatcher;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.regex.Pattern;
import java.util.regex.PatternSyntaxException;
import java.util.stream.Collectors;

import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.Multimap;
import io.netty.util.internal.ConcurrentSet;
import org.junit.After;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;

import ru.yandex.market.graphouse.search.tree.MetricNameZip;
import ru.yandex.misc.random.Random2;

public class MetricTreeTest {

    private MetricTree tree;
    private ExecutorService executor;

    @Before
    public void init() {
        tree = new MetricTree(new StockpileIdGenForTests(), new StockpileRetentionManagerForTests());
        executor = Executors.newFixedThreadPool(16);
    }

    @After
    public void shutdown() {
        executor.shutdownNow();
    }

    public static Pattern createPattern(final String globPattern) {
        String result = globPattern.replace("*", "[-_0-9a-zA-Z]*");
        result = result.replace("?", "[-_0-9a-zA-Z]");
        try {
            return Pattern.compile(result);
        } catch (PatternSyntaxException e) {
            return null;
        }
    }

    @Test
    public void testGlob() {
        Multimap<String, String> pattern2Candidates = generate();
        for (Map.Entry<String, Collection<String>> pattern2CandidatesMap : pattern2Candidates.asMap().entrySet()) {
            String glob = pattern2CandidatesMap.getKey();
            Pattern pattern = createPattern(glob);
            if (pattern == null) {
                System.out.println("Wrong pattern " + glob);
                continue;
            }
            for (String node : pattern2CandidatesMap.getValue()) {
                System.out.println(String.format("%40s\t%40s\t%s", glob, node, pattern.matcher(node).matches()));
            }
        }
    }

    @Test
    public void testGlobPath() {
        PathMatcher matcher = MetricTree.createPathMatcher("asdf[");
        Assert.assertNull(matcher);

        Multimap<String, String> pattern2Candidates = generate();
        for (Map.Entry<String, Collection<String>> pattern2CandidatesMap : pattern2Candidates.asMap().entrySet()) {
            String glob = pattern2CandidatesMap.getKey();
            matcher = MetricTree.createPathMatcher(glob);
            if (matcher == null) {
                System.out.println("Wrong pattern " + glob);
                continue;
            }
            for (String node : pattern2CandidatesMap.getValue()) {
                System.out.println(String.format("%40s\t%40s\t%s", glob, node, MetricTree.matches(matcher, node)));
            }
        }
    }

    private Multimap<String, String> generate() {
        Multimap<String, String> pattern2Candidates = ArrayListMultimap.create();
        pattern2Candidates.putAll("msh0[1-6]d_market_yandex_net", Arrays.asList("msh01d_market_yandex_net", "msh03d_market_yandex_net"));
        pattern2Candidates.putAll("min.market-front*.e", Arrays.asList("min.market-front.e", "min.market-front-ugr.e"));
        pattern2Candidates.putAll("min.market-front{-ugr,-fol}.e", Arrays.asList("min.market-front-fol.e", "min.market-front-ugr.e"));
        pattern2Candidates.putAll("min.market-front{,-ugr,-fol}.e", Arrays.asList("min.market-front.e", "min.market-front-ugr.e"));
        return pattern2Candidates;
    }

    @Test
    public void testContainsExpression() throws Exception {
        Assert.assertTrue(MetricTree.containsExpressions("msh0[1-6]d_market_yandex_net"));
    }

    @Test
    public void testSearch() throws Exception {
        tree.add("five_sec.int_8742.x1");
        tree.add("five_sec.int_8742.x1");
        tree.add("five_sec.int_8743.x1");
        tree.add("five_sec.int_8742.x2");

        search("five_sec.int_874?.x1", "five_sec.int_8742.x1", "five_sec.int_8743.x1");
        search("five_sec.int_8742.x*", "five_sec.int_8742.x1", "five_sec.int_8742.x2");
        search("*", "five_sec.");
        search("five_sec.*", "five_sec.int_8742.", "five_sec.int_8743.");
    }

    @Test
    public void testFindDoesNotCreate() throws Exception {
        tree.add("five_sec.int_8742.x1");

        tree.findExistingMetric("G A R.B A.G E");
        search("G A R.", "");
    }

    @Test
    public void testStatusesWorkflow() throws Exception {
        Assert.assertEquals(MetricTreeStatus.SIMPLE, tree.add("five_sec.int_8742.x1").getStatus());
        Assert.assertEquals(MetricTreeStatus.SIMPLE, tree.add("five_sec.int_8742.x1").getStatus());

        // BAN -> APPROVED
        tree.add("five_sec.int_8743.x1");
        Assert.assertEquals(MetricTreeStatus.BAN, tree.modify("five_sec.int_8743.", MetricTreeStatus.BAN).getStatus());
        searchWithMessage("Dir is BANned, but we found it", "five_sec.*", "five_sec.int_8742.");
        searchWithMessage("Dir is BANned, but we found it's metric", "five_sec.int_8743.", "");
        assertThrows("Dir is BANned, but we can add metric into it", MetricResponseStatus.REJECTED_BANNED, () -> tree.add("five_sec.int_8743.x0"));
        assertThrows("Dir is BANned, but we can add dir into it", MetricResponseStatus.REJECTED_BANNED, () -> tree.add("five_sec.int_8743.new."));

        Assert.assertEquals(MetricTreeStatus.APPROVED, tree.modify("five_sec.int_8743.", MetricTreeStatus.APPROVED).getStatus());
        search("five_sec.*", "five_sec.int_8742.", "five_sec.int_8743.");

        // HIDDEN
        search("five_sec.int_8742.*", "five_sec.int_8742.x1");
        Assert.assertEquals(MetricTreeStatus.HIDDEN, tree.modify("five_sec.int_8742.", MetricTreeStatus.HIDDEN).getStatus());
        searchWithMessage("Dir is HIDDEN, but we found it", "five_sec.*", "five_sec.int_8743.");
        searchWithMessage("Dir is HIDDEN, but we found it's metric", "five_sec.int_8742.*", "");
        Assert.assertEquals(MetricTreeStatus.SIMPLE, tree.add("five_sec.int_8742.x2").getStatus());
        search("five_sec.int_8742.*", "five_sec.int_8742.x1", "five_sec.int_8742.x2");
        Assert.assertEquals(MetricTreeStatus.APPROVED, tree.modify("five_sec.int_8742.", MetricTreeStatus.APPROVED).getStatus());
        search("five_sec.*", "five_sec.int_8742.", "five_sec.int_8743.");

        // SIMPLE -> AUTO_HIDDEN -> SIMPLE
        search("five_sec.int_8742.*", "five_sec.int_8742.x1", "five_sec.int_8742.x2");
        Assert.assertEquals(MetricTreeStatus.HIDDEN, tree.modify("five_sec.int_8742.x2", MetricTreeStatus.HIDDEN).getStatus());
        searchWithMessage("Metric is HIDDEN, but we found it", "five_sec.int_8742.*", "five_sec.int_8742.x1");
        Assert.assertEquals(MetricTreeStatus.HIDDEN, tree.modify("five_sec.int_8742.x1", MetricTreeStatus.HIDDEN).getStatus());
        searchWithMessage("Dir is AUTO_HIDDEN, but we found it", "five_sec.*", "five_sec.int_8743.", "five_sec.int_8742."); //Cause "five_sec.int_8742." is Approved
        Assert.assertEquals(MetricTreeStatus.SIMPLE, tree.add("five_sec.int_8742.x3").getStatus());
        searchWithMessage("We added new metric in AUTO_HIDDEN dir, but dir is still AUTO_HIDDEN",
            "five_sec.*", "five_sec.int_8742.", "five_sec.int_8743.");
        search("five_sec.int_8742.*", "five_sec.int_8742.x3");

        Assert.assertEquals(MetricTreeStatus.SIMPLE, tree.add("five_sec.int_8742.x2.y1").getStatus());
        searchWithMessage("We added new metric, but dir is still AUTO_HIDDEN",
            "five_sec.*", "five_sec.int_8742.", "five_sec.int_8743.");
        search("five_sec.int_8742.*", "five_sec.int_8742.x2.", "five_sec.int_8742.x3");
    }

    @Test
    public void testBanMetrics() throws MetricAddException, IOException {
        List<String> metrics = Arrays.asList(
            "q.w.e1.r1",
            "q.w.e1.r2",
            "q.w.e1.r3",
            "q.w.e2.r1");
        for (String metric : metrics) {
            tree.add(metric);
        }
        searchAllMetrics("q.*", metrics.toArray(new String[metrics.size()]));
        searchAllMetrics("q.w.*", metrics.toArray(new String[metrics.size()]));
        searchAllMetrics("q.w.e1.*", metrics.subList(0, 3).toArray(new String[metrics.size() - 1]));
        searchAllMetrics("q.w.e2.*", metrics.subList(3, 4).toArray(new String[1]));

        tree.modify("q.w.e1.r1", MetricTreeStatus.BAN);
        // if you ban one metric then read returns metric with ban status
        MetricNameZip bannedMetric = tree.findExistingMetric("q.w.e1.r1");
        Assert.assertEquals(MetricTreeStatus.BAN, bannedMetric.getStatus());
        searchWithMessage("should be empty, because user can't read banned metrics", "q.w.e1.r1", "");

        // the rest metrics should exist in the tree
        for (int i = 1; i < metrics.size(); i++) {
            String currentMetric = metrics.get(i);
            MetricNameZip metricFromTree = tree.findExistingMetric(currentMetric);
            Assert.assertEquals(currentMetric, metricFromTree.getName());
        }
        searchAllMetrics("q.w.*", metrics.subList(1, 4).toArray(new String[metrics.size() - 1]));

        // if you banned Dir then read returns null,
        // because findExistingMetric catch MetricAddException and return null
        tree.modify("q.w.e1.", MetricTreeStatus.BAN);
        for (int i = 0; i < 3; i++) {
            String currentMetric = metrics.get(i);
            bannedMetric = tree.findExistingMetric(currentMetric);
            Assert.assertNull(bannedMetric);
            searchWithMessage("should be empty, because user can't read banned metrics", currentMetric, "");
        }
        MetricNameZip metricFromTree = tree.findExistingMetric(metrics.get(3));
        Assert.assertEquals(metrics.get(3), metricFromTree.getName());
    }

    @Test
    public void testBanMetricsConcurrency() throws ExecutionException, InterruptedException, IOException {
        int concurrentTasks = 32;
        Set<String> metrics = new ConcurrentSet<>();
        Runnable insertTask = () -> {
            Set<String> localMetrics = generateRandomMetrics(1000);
            metrics.addAll(localMetrics);
            for (String metric : localMetrics) {
                try {
                    tree.add(metric);
                } catch (MetricAddException e) {
                    Assert.fail();
                }
            }
        };
        Future[] futures = new Future[concurrentTasks];
        for (int i = 0; i < concurrentTasks; i++) {
            futures[i] = executor.submit(insertTask);
        }
        for (int i = 0; i < concurrentTasks; i++) {
            futures[i].get();
        }
        List<String> metricsList = new ArrayList<>(metrics);
        List<HashSet<String>> sets = new ArrayList<>();
        int deletedMetricsPerThread = 700;

        for (int j = 0; j < concurrentTasks; j++) {
            sets.add(new HashSet<>());
            for (int i = 0; i < deletedMetricsPerThread; i++) {
                int index = Random2.threadLocal().nextInt(metricsList.size());
                String bannedMetric = metricsList.get(index);
                sets.get(j).add(bannedMetric);
                metricsList.remove(index);
            }
        }

        int expectedDeletedCount = concurrentTasks * deletedMetricsPerThread;
        Set<String> bannedMetrics = new ConcurrentSet<>();
        AtomicInteger threadId = new AtomicInteger(0);
        Runnable deletion = () -> {
            int setId = threadId.getAndIncrement();
            try {
                for (String metric : sets.get(setId)) {
                    tree.modify(metric, MetricTreeStatus.BAN);
                    bannedMetrics.add(metric);
                }
            } catch (MetricAddException e) {
                Assert.fail();
            }
        };

        futures = new Future[concurrentTasks];
        for (int i = 0; i < concurrentTasks; i++) {
            futures[i] = executor.submit(deletion);
        }
        for (int i = 0; i < concurrentTasks; i++) {
            futures[i].get();
        }
        for (String existedMetric : metricsList) {
            MetricNameZip metricFromTree = tree.findExistingMetric(existedMetric);
            Assert.assertEquals(existedMetric, metricFromTree.getName());
            Assert.assertEquals(MetricTreeStatus.SIMPLE, metricFromTree.getStatus());
        }
        for (String bannedMetric : bannedMetrics) {
            MetricNameZip metricFromTree = tree.findExistingMetric(bannedMetric);
            Assert.assertEquals(bannedMetric, metricFromTree.getName());
            Assert.assertEquals(MetricTreeStatus.BAN, metricFromTree.getStatus());
        }
        Assert.assertEquals(expectedDeletedCount, bannedMetrics.size());
        metrics.removeAll(bannedMetrics);
        searchAllMetrics("*", metrics.toArray(new String[metrics.size()]));
    }

    @Test
    public void banMetricsWithAsterisk() throws MetricAddException, IOException {
        Set<String> metrics = generateRandomMetrics(5000);
        for (String metric : metrics) {
            tree.add(metric);
        }
        ArrayList<String> metricsList = new ArrayList<>(metrics);
        Random2 random2 = Random2.threadLocal();
        String[] split;
        do {
            int index = random2.nextInt(metricsList.size());
            String banMetric = metricsList.get(index);
            split = banMetric.split("\\.");
        } while (split.length < 3);

        List<String> nodes = Arrays.asList(split).subList(0, 2);
        String banMetric = "";
        for (String node : nodes) {
            banMetric += node + ".";
        }
        String regexForPattern = banMetric.replaceAll("\\.", "\\\\.") + ".+";
        Pattern pattern = Pattern.compile(regexForPattern);
        List<String> bannedMetrics =
            metricsList.stream()
                .filter(e -> pattern.matcher(e).matches())
                .collect(Collectors.toList());
        ArrayList<String> simpleMetrics = new ArrayList<>(metricsList);
        for (String bannedMetric : bannedMetrics) {
            simpleMetrics.remove(bannedMetric);
        }
        tree.modify(banMetric, MetricTreeStatus.BAN);
        for (String existedMetric : simpleMetrics) {
            MetricNameZip metricFromTree = tree.findExistingMetric(existedMetric);
            Assert.assertEquals(existedMetric, metricFromTree.getName());
            Assert.assertEquals(MetricTreeStatus.SIMPLE, metricFromTree.getStatus());
        }
        for (String bannedMetric : bannedMetrics) {
            MetricNameZip metricFromTree = tree.findExistingMetric(bannedMetric);
            Assert.assertNull(metricFromTree);
        }
        searchAllMetrics("*", simpleMetrics.toArray(new String[simpleMetrics.size()]));
    }

    private Set<String> generateRandomMetrics(int n) {
        Set<String> metrics = new HashSet<>();
        Random2 random2 = Random2.threadLocal();
        int i = 0;
        do {
            int length = random2.nextInt(2) + 1;
            // can't change MetricTreeStatus of the root or son of the root
            int nodesCount = random2.nextInt(8) + 2;
            StringBuilder metric = new StringBuilder();
            for (int j = 0; j < nodesCount; j++) {
                metric.append(random2.nextString(length, "abcde"));
                if (j < nodesCount - 1) {
                    metric.append(".");
                }
            }
            boolean hasPrefix = false;
            for (String s : metrics) {
                if (s.startsWith(metric.toString()) || metric.toString().startsWith(s)) {
                    hasPrefix = true;
                    break;
                }
            }
            if (hasPrefix) {
                continue;
            }
            boolean added = metrics.add(metric.toString());
            if (added) {
                i++;
            }
        } while (i < n);
        return metrics;
    }

    private void search(String pattern, String... expected) throws IOException {
        searchWithMessage("", pattern, expected);
    }

    private void searchAllMetrics(String pattern, String... expected) throws IOException {
        searchAllMetricsWithMessage("", pattern, expected);
    }

    private void searchWithMessage(String message, String pattern, String... expected) throws IOException {
        Arrays.sort(expected);
        StringBuilder result = new StringBuilder();
        tree.search(pattern, result);
        String[] actual = result.toString().split("\\n");
        Arrays.sort(actual);
        Assert.assertArrayEquals(message + "\nFound " + result, expected, actual);
    }

    private void searchAllMetricsWithMessage(String message, String pattern, String... expected) {
        Arrays.sort(expected);
        List<String> actual = new ArrayList<>();
        tree.searchAllMetrics(pattern, actual::add);
        Collections.sort(actual);
        Assert.assertArrayEquals(message + "\nFound ", expected, actual.toArray());
    }

    private void assertThrows(String message, MetricResponseStatus status, RunnableWithException<MetricAddException> code) {
        try {
            code.run();
            Assert.fail(message);
        } catch (MetricAddException e) {
            Assert.assertEquals(status, e.status);
        }
    }

    interface RunnableWithException<E extends Exception> {
        void run() throws E;
    }
}
