package ru.yandex.market.graphouse.search.dao;

import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadLocalRandom;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import java.util.stream.Stream;

import org.apache.commons.lang3.RandomStringUtils;
import org.junit.Before;
import org.junit.Test;

import ru.yandex.market.graphouse.search.MetricDbRow;
import ru.yandex.stockpile.client.shard.StockpileLocalId;
import ru.yandex.stockpile.client.shard.StockpileShardId;

import static org.junit.Assert.assertArrayEquals;
import static org.junit.Assert.assertEquals;

/**
 * @author Vladimir Gordiychuk
 */
public abstract class AbstractMetricsDaoTest {

    private Set<String> usedMetricNames;

    @Before
    public void setUp() throws Exception {
        usedMetricNames = new HashSet<>();
    }

    @Test
    public void testLoadAllMetrics() {
        var expected = insert(500, Instant.now()).stream()
                .map(MetricRow::mapToMetricDbRow)
                .sorted()
                .toArray(MetricDbRow[]::new);

        var loaded = loadAllMetrics().toArray(MetricDbRow[]::new);
        assertArrayEquals(expected, loaded);
    }

    @Test
    public void testLoadManyMetrics() {
        var now = Instant.now();
        var rows = IntStream.range(0, 10_000)
                .mapToObj(id -> MetricRow.of(String.format("%017d", id), 0, now, StockpileShardId.random(1000), StockpileLocalId.random()))
                .collect(Collectors.toList());
        getDao().saveMetrics(rows).join();

        var expected = rows.stream()
                .map(MetricRow::mapToMetricDbRow)
                .sorted()
                .toArray(MetricDbRow[]::new);

        var loaded = loadAllMetrics().toArray(MetricDbRow[]::new);
        assertArrayEquals(expected, loaded);

        var fresh = loadNewMetrics(Instant.EPOCH).toArray(MetricDbRow[]::new);
        assertArrayEquals(expected, fresh);
    }

    @Test
    public void testSaveMetricOnlyOnce() {
        var saved = MetricRow.of("alice", 0, Instant.now().minusSeconds(1_000), 42, 123);
        getDao().saveMetrics(List.of(saved)).join();

        var ignore = MetricRow.of("alice", 0, Instant.now(), 33, 5555);
        getDao().saveMetrics(List.of(ignore)).join();

        assertEquals(List.of(saved.mapToMetricDbRow()), loadAllMetrics());
        assertEquals(List.of(saved.mapToMetricDbRow()), loadNewMetrics(Instant.EPOCH));
    }

    @Test
    public void testSaveMetrics() {
        var now = Instant.now().minusSeconds(10);
        var expected = insert(3_000, now).stream()
                .map(MetricRow::mapToMetricDbRow)
                .sorted()
                .toArray(MetricDbRow[]::new);

        var allMetrics = loadAllMetrics().toArray(MetricDbRow[]::new);
        var newMetrics = loadNewMetrics(now.minusSeconds(30)).toArray(MetricDbRow[]::new);
        assertArrayEquals(expected, allMetrics);
        assertArrayEquals(expected, newMetrics);
    }

    @Test
    public void testDeleteMetrics() {
        var inserted = insert(5_000, Instant.now().minusSeconds(10));
        var toRemove = new ArrayList<String>();
        var expected = new ArrayList<MetricDbRow>();
        var random = ThreadLocalRandom.current();
        for (var row : inserted) {
            if (random.nextBoolean()) {
                expected.add(row.mapToMetricDbRow());
            } else {
                toRemove.add(row.name());
            }
        }
        expected.sort(MetricDbRow::compareTo);
        getDao().remove(toRemove).join();

        var loaded = loadAllMetrics();
        assertArrayEquals(expected.toArray(), loaded.toArray());
    }

    @Test
    public void testReadNewMetrics() {
        var before = insert(300, Instant.now().minusSeconds(1000))
                .stream()
                .map(MetricRow::mapToMetricDbRow)
                .sorted()
                .toArray(MetricDbRow[]::new);

        var after = insert(50, Instant.now().plusSeconds(1000))
                .stream()
                .map(MetricRow::mapToMetricDbRow)
                .sorted()
                .toArray(MetricDbRow[]::new);

        var all = Stream.concat(Stream.of(before), Stream.of(after))
                .sorted()
                .toArray();

        var allMetrics = loadAllMetrics();
        assertArrayEquals(all, allMetrics.toArray());

        var newMetrics = loadNewMetrics(Instant.now());
        assertArrayEquals(after, newMetrics.toArray());
    }

    @Test
    public void testRemoveOld() {
        var now = Instant.now();
        var old = insert(500, now.minus(30, ChronoUnit.DAYS))
                .stream()
                .map(MetricRow::mapToMetricDbRow)
                .sorted()
                .toArray(MetricDbRow[]::new);

        var fresh = insert(100, now)
                .stream()
                .map(MetricRow::mapToMetricDbRow)
                .sorted()
                .toArray(MetricDbRow[]::new);

        var all = Stream.concat(Stream.of(old), Stream.of(fresh))
                .sorted()
                .toArray();

        var loadedAll = loadNewMetrics(Instant.EPOCH);
        assertArrayEquals(all, loadedAll.toArray());

        getDao().removeOldRows().join();

        var loadedFresh = loadNewMetrics(Instant.EPOCH);
        assertArrayEquals(fresh, loadedFresh.toArray());
    }

    protected abstract MetricsDao getDao();

    private List<MetricRow> insert(int n, Instant instant) {
        List<MetricRow> result = new ArrayList<>();
        while (result.size() < n) {
            String name = RandomStringUtils.randomAlphabetic(15);
            if (!usedMetricNames.add(name)) {
                continue;
            }

            result.add(MetricRow.of(name, 0, instant, StockpileShardId.random(1000), StockpileLocalId.random()));
        }
        getDao().saveMetrics(result).join();
        return result;
    }

    private List<MetricDbRow> loadAllMetrics() {
        BlockingQueue<MetricDbRow> rows = new LinkedBlockingQueue<>();
        int size = getDao().loadAllMetrics(batch -> {
            for (MetricDbRow row : batch) {
                rows.add(row);
            }
        }).join();
        assertEquals(rows.size(), size);
        return rows.stream()
                .sorted()
                .collect(Collectors.toList());
    }

    private List<MetricDbRow> loadNewMetrics(Instant from) {
        BlockingQueue<MetricDbRow> rows = new LinkedBlockingQueue<>();
        int size = getDao().loadNewMetrics(batch -> {
            for (MetricDbRow row : batch) {
                rows.add(row);
            }
        }, Math.toIntExact(from.getEpochSecond())).join();
        assertEquals(rows.size(), size);
        return rows.stream()
                .sorted()
                .collect(Collectors.toList());
    }
}
