package ru.yandex.market.graphouse.search.util;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.locks.ReentrantLock;

import javax.annotation.Nonnull;

import org.junit.Assert;
import org.junit.Test;

import ru.yandex.misc.lang.number.LongUtils;

/**
 * @author Maksim Leonov (nohttp@)
 */
public class ConcurrentExpandingMapTest {

    private Collection<ConcurrentExpandingMapBase<String, String, ?>> makeMaps() {
        return List.of(
            new ConcurrentExpandingMapToIndex<>() {
                private final ArrayList<String> storage = new ArrayList<>();

                @Override
                public String indexToValueImpl(long idx) {
                    return storage.get(LongUtils.toIntExact(idx));
                }

                @Override
                public long valueToIndexImpl(String val) {
                    storage.add(val);
                    return storage.size() - 1;
                }

                @Override
                public String keyForValue(@Nonnull String value) {
                    return value;
                }

                @Override
                public int hash(@Nonnull String key) {
                    return key.hashCode();
                }
            }
        );
    }

    @Test
    public void testPut() {
        for (ConcurrentExpandingMapBase<String, String, ?> map : makeMaps()) {
            ReentrantLock lock = new ReentrantLock();

            String aa1 = "aa";
            String aa2 = new String("aa");
            String ab = "ab";

            Assert.assertNull(map.putIfAbsent(aa1, lock));
            Assert.assertNull(map.putIfAbsent(ab, lock));
            Assert.assertSame(aa1, map.putIfAbsent(aa2, lock));
            Assert.assertSame(aa1, map.get(aa1));
        }
    }

    @Test
    public void testGet() {
        for (ConcurrentExpandingMapBase<String, String, ?> map : makeMaps()) {
            ReentrantLock lock = new ReentrantLock();

            map.putIfAbsent("1", lock);
            map.putIfAbsent("2", lock);

            Assert.assertEquals("1", map.get("1"));
            Assert.assertEquals("2", map.get("2"));
            Assert.assertNull(map.get("3"));
        }
    }

    @Test
    public void testValues() {
        for (ConcurrentExpandingMapBase<String, String, ?> map : makeMaps()) {
            ReentrantLock lock = new ReentrantLock();

            Map<String, String> checker = new HashMap<>();
            for (char first = 'a'; first < 'z'; first++) {
                String s = new String(new char[]{first});
                map.putIfAbsent(s, lock);
                checker.put(s, s);
            }

            assertSame(map, checker);

            Assert.assertNull(map.get("zzz"));
        }
    }

    @Test
    public void testLoad() {
        for (ConcurrentExpandingMapBase<String, String, ?> map : makeMaps()) {
            ReentrantLock lock = new ReentrantLock();

            Map<String, String> checker = new HashMap<>();
            for (char first = 'a'; first < 'z'; first++) {
                for (char second = 'a'; second < 'z'; second++) {
                    char[] chars = new char[]{first, second};
                    String s = new String(chars);

                    map.putIfAbsent(s, lock);
                    checker.put(s, s);
                }
            }

            assertSame(map, checker);

            Assert.assertNull(map.get("zzz"));
        }
    }

    private <V> void assertSame(ConcurrentExpandingMapBase<V, V, ?> cem, Map<V, V> chm) {
        for (V s : cem.values()) {
            Assert.assertSame(s, chm.get(s));
        }

        for (V s : chm.values()) {
            Assert.assertSame(s, cem.get(s));
        }
    }
}
