package ru.yandex.market.graphouse.server;

import java.io.IOException;

import org.junit.Assert;
import org.junit.Test;

import ru.yandex.market.graphouse.search.MetricResponseStatus;

/**
 * @author Dmitry Andreev <a href="mailto:AndreevDm@yandex-team.ru"/>
 * @date 03/06/15
 */
public class MetricValidatorTest {

    private static final String IP = "127.0.0.1";

    @Test
    public void testValidate() {
        invalid("-嘊-嘊嘍-嘍-aaa_.._tt_personal-billing-report_xml.0_995");
        invalid("market.mbo-front-iva.timings-dynamic.3febfdd52d4fea02xxx'x22<>_.._tt_personal-billing-report_xml_.0_995");
        invalid("market.mbo-front-iva.timings-dynamic.tt_personal-billing-report_xml_3febfdd52d4fea02xxx'x22<>_...0_99");

        invalid("gdsgsgs");
        invalid("one_min.fdsfdsfs..fdsfsfsd");
        invalid("one_min.fdsfdsfs.fdsfsfsd.");
        invalid(".one_min.fdsfdsfs.fdsfsfsd");
        invalid("one_min..x");
        invalid("a.b");
        invalid("e1.e2.e3.e4.e5.e6.e7.e8.e9.e10.e11.e12.e13.e14.e15.e16.e17");
        valid("one_min.fdsfdsfs.fdsfsfsd");
        valid("one_min.fdsfdsfs.fdsfsf(sd)");
        valid("one_min.fdsfdsfs.fdsfsfsd");
        valid("one_min.fdsfdsfs.fdsfsf[sd]");
    }

    @Test
    public void testBlacklist() throws IOException {
        MetricValidator validator = new MetricValidator();
        try {
            validator.addBlacklistPattern("aaa\\.bbb\\.ccc\\.ddd.*");
            validator.addBlacklistPattern("aaa\\.bbb\\.ccc\\.ddd\\.eee\\..*");
            validator.addBlacklistPattern("aaa\\.bbb\\..*\\.e");
            assertBlacklist(validator, "aaa.bbb.ccc.ddd");
            assertBlacklist(validator, "aaa.bbb.ccc.dddq");
            assertBlacklist(validator, "aaa.bbb.ccc.ddd.q1");
            assertBlacklist(validator, "aaa.bbb.ccc.ddd.eee.e");
            assertBlacklist(validator, "aaa.bbb.ccc.ddd.eee.t1");
            assertBlacklist(validator, "aaa.bbb.ccc.ddd.eee.t1.t2");
            assertBlacklist(validator, "aaa.bbb.qqqq.e");
            assertBlacklist(validator, "aaa.bbb.QWE123.e");

            assertValid(validator, "aaa.bbb.ccc.dde");
            assertValid(validator, "aaa.bbb.ccc.dd");
            assertValid(validator, "aaa.bbb.ccc.d");
            assertValid(validator, "aaa.bbb.QWE123.e1");
        } finally {
            validator.clearFile();
        }
    }

    private void valid(String metric) {
        assertValid(MetricValidator.DEFAULT, metric);
    }

    private void invalid(String metric) {
        Assert.assertNotSame(MetricResponseStatus.OK, MetricValidator.DEFAULT.validate(metric, IP));
    }

    private void assertBlacklist(MetricValidator validator, String metric) {
        Assert.assertSame(MetricResponseStatus.BLACKLIST_METRIC, validator.validate(metric, IP));
    }

    private void assertValid(MetricValidator validator, String metric) {
        Assert.assertSame(MetricResponseStatus.OK, validator.validate(metric, IP));
    }
}
