package ru.yandex.market.graphouse.stockpile.proxy;

import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ForkJoinPool;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;

import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;

import ru.yandex.solomon.metrics.client.StockpileClientStub;
import ru.yandex.solomon.model.point.AggrPoint;
import ru.yandex.solomon.model.point.column.StockpileColumns;
import ru.yandex.solomon.model.point.column.TsColumn;
import ru.yandex.solomon.model.point.column.ValueColumn;
import ru.yandex.solomon.model.protobuf.MetricId;
import ru.yandex.solomon.model.timeseries.AggrGraphDataArrayList;
import ru.yandex.stockpile.api.EStockpileStatusCode;
import ru.yandex.stockpile.client.shard.StockpileMetricId;

import static java.util.concurrent.CompletableFuture.completedFuture;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;
import static ru.yandex.solomon.model.point.AggrPointDataTestSupport.randomPoint;

/**
 * @author Vladimir Gordiychuk
 */
public class CrossDcStockpileClientTest {

    private Dc alice;
    private Dc bob;

    private CrossDcStockpileClient client;

    @Before
    public void setUp() {
        alice = new Dc("alice");
        bob = new Dc("bob");

        client = new CrossDcStockpileClient(alice.client, bob.client);
    }

    @Test
    public void combineSamePoints() {
        var mask = TsColumn.mask | ValueColumn.mask;
        var metric = StockpileMetricId.random();
        var ts0 = System.currentTimeMillis();

        AggrGraphDataArrayList source = new AggrGraphDataArrayList();
        AggrPoint point = new AggrPoint();
        for (int index = 0; index < 10; index++) {
            randomPoint(point, mask);
            point.tsMillis = ts0 + (index * 10_000);
            source.addRecord(point);
        }

        alice.addTimeseries(metric, source);
        bob.addTimeseries(metric, source);

        var response = client.readOne(ReadRequest.newBuilder()
            .setKey(metric)
            .build())
            .join();

        assertTrue(response.isOk());
        assertEquals(source, AggrGraphDataArrayList.of(response.getSource()));
    }

    @Test
    public void combineGaps() {
        var mask = TsColumn.mask | ValueColumn.mask;
        var metric = StockpileMetricId.random();
        var ts0 = System.currentTimeMillis();

        AggrGraphDataArrayList expected = new AggrGraphDataArrayList();
        AggrGraphDataArrayList alicePoints = new AggrGraphDataArrayList();
        AggrGraphDataArrayList bobPoints = new AggrGraphDataArrayList();

        AggrPoint point = new AggrPoint();
        ThreadLocalRandom random = ThreadLocalRandom.current();
        for (int index = 0; index < 100; index++) {
            randomPoint(point, mask, random);
            point.tsMillis = ts0 + (index * 10_000);
            expected.addRecord(point);

            switch (random.nextInt(3)) {
                case 0:
                    alicePoints.addRecord(point);
                    break;
                case 1:
                    bobPoints.addRecord(point);
                    break;
                case 2:
                    alicePoints.addRecord(point);
                    bobPoints.addRecord(point);
                    break;
            }
        }

        alice.addTimeseries(metric, alicePoints);
        bob.addTimeseries(metric, bobPoints);

        var response = client.readOne(ReadRequest.newBuilder()
            .setKey(metric)
            .build())
            .join();

        assertTrue(response.isOk());
        assertEquals(expected, AggrGraphDataArrayList.of(response.getSource()));
    }

    @Test
    public void anyOrAllOf() {
        var metricId = StockpileMetricId.random();
        var point = randomPoint();
        var type = StockpileColumns.typeByMask(point.columnSet);
        ReadResponse ok = new ReadResponse(metricId, type, AggrGraphDataArrayList.of(point));
        ReadResponse fail = new ReadResponse(metricId, EStockpileStatusCode.DEADLINE_EXCEEDED, "");

        // OK
        {
            CompletableFuture<List<ReadResponse>> responsesFuture = CrossDcStockpileClient
                .anyOrAllOf(List.of(completedFuture(ok)), 10_000);
            // immediately completed future
            assertTrue(responsesFuture.isDone());
            Assert.assertEquals(List.of(ok), responsesFuture.join());
        }

        // FAIL
        {
            CompletableFuture<List<ReadResponse>> responsesFuture = CrossDcStockpileClient
                .anyOrAllOf(List.of(completedFuture(fail)), 10_000);
            // immediately completed future
            assertTrue(responsesFuture.isDone());
            Assert.assertEquals(List.of(fail), responsesFuture.join());
        }

        // OK, FAIL
        {
            CompletableFuture<List<ReadResponse>> responsesFuture = CrossDcStockpileClient.anyOrAllOf(List.of(
                completedFuture(ok),
                completedFuture(fail)),
                10_000);
            // immediately completed future
            assertTrue(responsesFuture.isDone());
            Assert.assertEquals(List.of(ok, fail), responsesFuture.join());
        }

        // OK, FAIL (delayed < maxTime)
        {
            List<ReadResponse> responses = CrossDcStockpileClient.anyOrAllOf(List.of(
                delayedResponse(fail, 1_000),
                completedFuture(ok)),
                3_000).join();
            Assert.assertEquals(List.of(fail, ok), responses);
        }

        // OK, FAIL (delayed > maxTime)
        {
            List<ReadResponse> responses = CrossDcStockpileClient.anyOrAllOf(List.of(
                delayedResponse(fail, 3_000),
                completedFuture(ok)),
                1_000).join();
            Assert.assertEquals(List.of(ok), responses);
        }

        // OK (delayed > maxTime), FAIL
        {
            List<ReadResponse> responses = CrossDcStockpileClient.anyOrAllOf(List.of(
                delayedResponse(ok, 3_000),
                completedFuture(fail)),
                1_000).join();
            // anyway we await OK response
            Assert.assertEquals(List.of(ok, fail), responses);
        }
    }

    private static CompletableFuture<ReadResponse> delayedResponse(ReadResponse r, long delayMillis) {
        CompletableFuture<ReadResponse> f = new CompletableFuture<>();
        CompletableFuture.delayedExecutor(delayMillis, TimeUnit.MILLISECONDS)
            .execute(() -> { f.complete(r); });
        return f;
    }

    private static class Dc {
        private String name;
        private StockpileClientStub stub;
        private DcStockpileClient client;

        public Dc(String name) {
            this.name = name;
            this.stub = new StockpileClientStub(ForkJoinPool.commonPool());
            this.client = new DcStockpileClient(stub);
        }

        public void addTimeseries(StockpileMetricId metricId, AggrGraphDataArrayList ts) {
            stub.addTimeSeries(MetricId.newBuilder()
                .setShardId(metricId.getShardId())
                .setLocalId(metricId.getLocalId())
                .build(), ts);
        }
    }
}
