package ru.yandex.market.graphouse.stockpile.proxy;

import java.util.concurrent.ForkJoinPool;

import org.junit.Before;
import org.junit.Test;

import ru.yandex.solomon.metrics.client.StockpileClientStub;
import ru.yandex.solomon.model.point.column.TsColumn;
import ru.yandex.solomon.model.point.column.ValueColumn;
import ru.yandex.solomon.model.protobuf.MetricId;
import ru.yandex.solomon.model.protobuf.MetricType;
import ru.yandex.solomon.model.timeseries.AggrGraphDataArrayList;
import ru.yandex.stockpile.client.shard.StockpileLocalId;
import ru.yandex.stockpile.client.shard.StockpileMetricId;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;
import static ru.yandex.solomon.model.point.AggrPointDataTestSupport.randomPoint;

/**
 * @author Vladimir Gordiychuk
 */
public class DcStockpileClientTest {

    private StockpileClientStub stub;
    private DcStockpileClient client;

    @Before
    public void setUp() {
        stub = new StockpileClientStub(ForkJoinPool.commonPool());
        client = new DcStockpileClient(stub);
    }

    @Test
    public void read() {
        var point = randomPoint(MetricType.DGAUGE);
        var shardId = stub.randomShardId();
        var localId = StockpileLocalId.random();

        stub.addTimeSeries(MetricId.newBuilder()
            .setShardId(shardId)
            .setLocalId(localId)
            .build(), AggrGraphDataArrayList.of(point));

        var response = client.readOne(ReadRequest.newBuilder()
            .setKey(new StockpileMetricId(shardId, localId))
            .build())
            .join();

        assertTrue(response.isOk());
        assertEquals(AggrGraphDataArrayList.of(point), AggrGraphDataArrayList.of(response.getSource()));
    }

    @Test
    public void readRange() {
        var ts0 = System.currentTimeMillis();
        var mask = TsColumn.mask | ValueColumn.mask;
        var shardId = stub.randomShardId();
        var localId = StockpileLocalId.random();

        AggrGraphDataArrayList source = new AggrGraphDataArrayList();
        for (int index = 0; index < 10; index++) {
            var point = randomPoint(mask);
            point.tsMillis = ts0 + (index * 10_000);
            source.addRecord(point);
        }

        stub.addTimeSeries(MetricId.newBuilder()
            .setShardId(shardId)
            .setLocalId(localId)
            .build(), source);

        var response = client.readOne(ReadRequest.newBuilder()
            .setKey(new StockpileMetricId(shardId, localId))
            .setFromMillis(ts0 + (3 * 10_000))
            .setToMillis(ts0 + (5 * 10_000))
            .build())
            .join();

        assertTrue(response.isOk());
        assertEquals(AggrGraphDataArrayList.of(source.slice(3, 5)), AggrGraphDataArrayList.of(response.getSource()));
    }
}
