#include "init.h"

#include <solomon/libs/cpp/host_resolver/host_resolver.h>
#include <solomon/libs/cpp/cluster_map/cluster.h>
#include <solomon/libs/cpp/dns/dns.h>

#include <library/cpp/actors/core/actorsystem.h>
#include <library/cpp/actors/core/log.h>

#include <library/cpp/actors/dnsresolver/dnsresolver.h>

#include <library/cpp/actors/interconnect/interconnect.h>
#include <library/cpp/actors/interconnect/interconnect_common.h>
#include <library/cpp/actors/interconnect/interconnect_tcp_proxy.h>
#include <library/cpp/actors/interconnect/interconnect_tcp_server.h>
#include <library/cpp/actors/interconnect/poller_actor.h>
#include <library/cpp/actors/interconnect/poller_tcp.h>
#include <library/cpp/monlib/metrics/metric_sub_registry.h>

#include <util/system/hostname.h>
#include <util/network/address.h>
#include <google/protobuf/repeated_field.h>

#include <utility>


using namespace NActors;
using namespace NActors::NDnsResolver;
using namespace NMonitoring;

namespace NSolomon {
namespace {
    // in actor system 0 means "this node", so we cannot use this id
    struct TIcNodeMapper: INodeMapper {
        ui32 GetId(TStringBuf hostname) const override {
            return Impl->GetId(hostname) + 1;
        }

        THolder<INodeMapper> Impl{CreateDefaultMapper()};
    };

    TInterconnectConfig FromAddressList(const NProtoBuf::RepeatedPtrField<TProtoStringType>& protoAddrs) {
        using namespace NThreading;
        TInterconnectConfig result;

        TVector<TStringBuf> addrs;
        Transform(protoAddrs.begin(), protoAddrs.end(), std::back_inserter(addrs), [] (auto&& s) {
            return TStringBuf{s};
        });

        TIcNodeMapper mapper;
        auto cluster = TClusterMapBase::Load(addrs, &mapper);
        const auto thisNode = cluster->Local();
        Y_ENSURE(!thisNode.IsUnknown(), "cannot determine on which cluster node we are");
        result.ThisNodeId = ui32(thisNode.NodeId);
        auto nodes = cluster->Nodes();
        auto dnsClient = CreateDnsClient();

        TVector<TIcEndpoint> endpoints;
        endpoints.resize(nodes.size());

        TVector<TFuture<void>> fs;
        size_t idx = 0;
        for (auto& node: nodes) {
            auto f = dnsClient->GetAddresses(node.Fqdn, true).Apply([idx, node, &endpoints] (auto fut) {
                TIpv6AddressesSet val;
                TString addr;

                try {
                    val = fut.ExtractValue();
                    addr = val.begin()->ToString(false);
                } catch (...) {
                    Cerr << "Failed to resolve address " << node.Fqdn
                         << ", skipping this node: " << CurrentExceptionMessage() << Endl;
                    return;
                }

                if (val.size() > 1) {
                    TStringBuilder sb;
                    sb << "Node " << node.Fqdn << " has multiple addresses. Will use " << addr << '\n';
                    Cerr << sb;
                }

                TIcEndpoint e{
                    .Hostname = node.Fqdn,
                    .Address = addr,
                    .Port = node.Port,
                    .NodeId = ui32(node.NodeId),
                };

                endpoints[idx] = std::move(e);
            }).IgnoreResult();

            fs.push_back(std::move(f));
            idx++;
        }

        auto all = WaitExceptionOrAll(fs);
        auto ok = all.Wait(TDuration::Minutes(1));
        Y_ENSURE(ok, "DNS resolution has timed out");
        all.TryRethrow();

        result.Endpoints = std::move(endpoints);

        return result;
    }

    TInterconnectConfig FromProtoConfig(const NProto::TInterconnectConfig& proto) {
        TInterconnectConfig result;

        if (proto.AddressesSize() > 0) {
            result = FromAddressList(proto.GetAddresses());
            return result;
        }
        // XXX can we avoid this?
        const auto thisHostname = FQDNHostName();

        for (auto&& node: proto.GetNodes()) {
            if (node.GetHostname() == thisHostname) {
                result.ThisNodeId = node.GetNodeId();
            }

            auto& endpoint = result.Endpoints.emplace_back();
            endpoint.Hostname = node.GetHostname();
            endpoint.Port = node.GetPort();
            endpoint.NodeId = node.GetNodeId();

            TNetworkAddress address(node.GetHostname(), node.GetPort());
            for (auto it = address.Begin(); it != address.End(); ++it) {
                if (it->ai_family != AF_INET6) {
                    continue;
                }

                endpoint.Address = NAddr::PrintHost(NAddr::TAddrInfo(&*it));
                break;
            }
            Y_ENSURE(endpoint.Address, "unable resolve IPv6 address for " + node.GetHostname());
        }

        return result;
    }
} // namespace

    void InitInterconnect(
        TActorSystemSetup& setup,
        TInterconnectConfig config,
        const std::shared_ptr<NMonitoring::TMetricRegistry>& metrics)
    {
        const auto nodeCount = config.Endpoints.size();

        Y_ENSURE(config.ThisNodeId != 0);
        Y_ENSURE(nodeCount > 0);

        setup.NodeId = config.ThisNodeId;

        setup.LocalServices.emplace_back(MakePollerActorId(), TActorSetupCmd{CreatePollerActor(), TMailboxType::ReadAsFilled, 0});

        auto nameserverTable = MakeIntrusive<TTableNameserverSetup>();

        TIcEndpoint* thisEndpoint{nullptr};
        ui32 maxNodeId = 0;

        for (auto&& endpoint: config.Endpoints) {
            if (endpoint.Hostname.empty()) { // we failed to resolve it earlier
                continue;
            }

            Y_ENSURE(endpoint.NodeId > 0);
            Y_ENSURE(endpoint.Port > 0);
            Y_ENSURE(endpoint.Address);
            Y_ENSURE(endpoint.Hostname);
            nameserverTable->StaticNodeTable[endpoint.NodeId] = std::make_pair(endpoint.Address, endpoint.Port);
            nameserverTable->StaticNodeTable[endpoint.NodeId].ResolveHost = endpoint.Hostname;

            maxNodeId = Max<ui32>(maxNodeId, endpoint.NodeId);

            if (endpoint.NodeId == config.ThisNodeId) {
                thisEndpoint = &endpoint;
            }
        }

        Y_ENSURE(thisEndpoint);

        setup.LocalServices.emplace_back(
            MakeDnsResolverActorId(),
            TActorSetupCmd{CreateOnDemandDnsResolver(), TMailboxType::ReadAsFilled, 0}
        );

        setup.LocalServices.emplace_back(
            GetNameserviceActorId(),
            TActorSetupCmd{CreateNameserverTable(nameserverTable), TMailboxType::ReadAsFilled, 0}
        );

        auto icCommon = MakeIntrusive<TInterconnectProxyCommon>();
        icCommon->NameserviceId = GetNameserviceActorId();
        icCommon->Metrics = metrics;
        icCommon->TechnicalSelfHostName = thisEndpoint->Address;
        icCommon->Settings.MergePerDataCenterCounters = true;
        icCommon->Settings.MergePerPeerCounters = true;

        // Taken from https://a.yandex-team.ru/arc/trunk/arcadia/kikimr/yf/actor_init/init.cpp?rev=r8836439#L295
        icCommon->ChannelsConfig.insert({0, {1}}); // 5% control
        icCommon->ChannelsConfig.insert({1, {19}}); // 95% data

        // TODO: support physical location

        setup.LocalServices.emplace_back(
            MakePollerActorId(),
            TActorSetupCmd{CreatePollerActor(), TMailboxType::ReadAsFilled, 0}
        );

        setup.Interconnect.ProxyActors.resize(maxNodeId + 1);

        for (auto&& endpoint: config.Endpoints) {
            if (endpoint.Hostname.empty()) { // we failed to resolve it earlier
                continue;
            }

            auto nodeId = endpoint.NodeId;

            if (nodeId != config.ThisNodeId) {
                auto* actor = new TInterconnectProxyTCP(nodeId, icCommon);
                setup.Interconnect.ProxyActors[nodeId] = TActorSetupCmd{actor, TMailboxType::ReadAsFilled, 0};
            } else {
                auto* listener = new TInterconnectListenerTCP{
                    endpoint.Address,
                    endpoint.Port,
                    icCommon
                };

                setup.LocalServices.emplace_back(
                    MakeInterconnectListenerActorId(false),
                    TActorSetupCmd{listener, TMailboxType::ReadAsFilled, 0}
                );
            }
        }
    }

    void InitInterconnect(
        NActors::TActorSystemSetup& setup,
        NProto::TInterconnectConfig protoConf,  // NOLINT(performance-unnecessary-value-param): false positive
        std::shared_ptr<NMonitoring::TMetricRegistry> metrics)  // NOLINT(performance-unnecessary-value-param): false positive
    {
        auto config = FromProtoConfig(std::move(protoConf));
        InitInterconnect(setup, std::move(config), std::move(metrics));
    }
} // namespace NSolomon
