#pragma once

#include <library/cpp/actors/core/actorid.h>
#include <library/cpp/actors/core/actorsystem.h>
#include <library/cpp/actors/core/hfunc.h>
#include <library/cpp/monlib/metrics/metric_registry.h>

#include <util/datetime/base.h>

#include <chrono>
#include <type_traits>

namespace NSolomon {

namespace NPrivate {

struct TTrackedEventTag {};

template <typename TEvent>
struct TMetrics {
    NMonitoring::IIntGauge* Queue = NMonitoring::TMetricRegistry::Instance()->IntGauge(
        {{"sensor", "actors.events.byType.queue"}, {"event", TypeName<TEvent>()}});
    NMonitoring::IRate* SendRate = NMonitoring::TMetricRegistry::Instance()->Rate(
        {{"sensor", "actors.events.byType.send"}, {"event", TypeName<TEvent>()}});
    NMonitoring::IRate* ReceiveRate = NMonitoring::TMetricRegistry::Instance()->Rate(
        {{"sensor", "actors.events.byType.receive"}, {"event", TypeName<TEvent>()}});
    NMonitoring::IHistogram* DeliveryTime = NMonitoring::TMetricRegistry::Instance()->HistogramRate(
        {{"sensor", "actors.events.byType.deliveryTimeUs"}, {"event", TypeName<TEvent>()}},
        NMonitoring::ExponentialHistogram(20, 2, 4));
};

} // namespace NPrivate

/**
 * An event with additional facilities for tracking and reporting.
 *
 * Actor system does not provide metrics detailed per event type, thus we have to implement them ourselves.
 * The idea is to record them when an event is sent or received. We record delivery time histogram,
 * number of sent and received events and number of queued events.
 *
 * This event inherits privately from `NActors::IEventBase` so that you can't bypass reporting statistics
 * by using the standard send facilities. Use `SendTracked` to send these events.
 *
 *
 * # Usage
 *
 * Derive your event from `TTrackedEvent`:
 *
 * ```
 * struct TEvent: public NSolomon::TTrackedEvent<TEvent, Event> {
 *     // ...
 * }
 * ```
 *
 * Use `SendTracked` to send your event. This will record an event being sent:
 *
 * ```
 * SendTracked(recipientId, MakeHolder<TEvent>());
 * ```
 *
 * When you receive an event, use `Receive` method on a handle to get an event from a handle and record an event
 * being delivered:
 *
 * ```
 * void OnEvent(TEvent::TPtr& handle) {
 *     auto event = handle->Receive();
 *     // ...
 * }
 * ```
 *
 * Note that handle does not mark event as received when it dies. If you didn't process an event,
 * it will stay inflight forever.
 */
template <typename TEvent, ui32 EventType_>
class TTrackedEvent: private NActors::IEventBase, public NPrivate::TTrackedEventTag {
public:
    static constexpr ui32 EventType = EventType_;

    class THandle: public NActors::IEventHandle {
    private:
        using NActors::IEventHandle::CastAsLocal;
        using NActors::IEventHandle::Get;
        using NActors::IEventHandle::Release;

    public:
        THandle(
                const NActors::TActorId& recipient,
                const NActors::TActorId& sender,
                THolder<TTrackedEvent> ev,
                ui32 flags = 0,
                ui64 cookie = 0)
            : NActors::IEventHandle(recipient, sender, static_cast<NActors::IEventBase*>(ev.Release()), flags, cookie)
        {
            Peek()->OnSend();
        }

    public:
        /**
         * Get access to the event without releasing it or marking it as received.
         */
        TEvent* Peek() {
            Y_ASSERT(Type == EventType);
            return static_cast<TEvent*>(GetBase());
        }

        /**
         * Mark event as received and release it from the handle.
         */
        THolder<TEvent> Receive() {
            THolder<TEvent> res = static_cast<TEvent*>(ReleaseBase().Release());
            res->OnReceive();
            return res;
        }
    };

    typedef TAutoPtr<THandle> TPtr;

public:
    TString ToStringHeader() const override {
        return TypeName<TEvent>();
    }

    TString ToString() const override {
        return TypeName<TEvent>();
    }

    ui32 Type() const override {
        return EventType;
    }

private:
    bool SerializeToArcadiaStream(NActors::TChunkSerializer* /*serializer*/) const override {
        Y_FAIL("Serialization of local event %s type %" PRIu32, TypeName<TEvent>().data(), EventType);
    }

    bool IsSerializable() const override {
        return false;
    }

public:
    TTrackedEvent() {
        static_assert(std::is_base_of_v<TTrackedEvent<TEvent, EventType>, TEvent>);
    }

public:
    /**
     * Record an event being send.
     */
    void OnSend() {
        SendTime_ = std::chrono::steady_clock::now();

        auto events = Singleton<NPrivate::TMetrics<TEvent>>();
        events->Queue->Inc();
        events->SendRate->Inc();
    }

    /**
     * Record an event being delivered.
     */
    void OnReceive() {
        auto receiveTime = std::chrono::steady_clock::now();

        auto events = Singleton<NPrivate::TMetrics<TEvent>>();
        events->Queue->Dec();
        events->ReceiveRate->Inc();
        events->DeliveryTime->Record(
            std::chrono::duration_cast<std::chrono::microseconds>(receiveTime - SendTime_).count());
    }

private:
    std::chrono::steady_clock::time_point SendTime_;
};

/**
 * Send `event` to `recipient` while updating tracking stats. Use this method instead of the normal send facilities
 * when working with `TTrackedEvent`-derived events.
 */
template <typename T>
inline void SendTracked(NActors::TActorSystem* system, NActors::TActorId recipient, NActors::TActorId sender,
                        THolder<T> event, ui32 flags = 0, ui64 cookie = 0)
{
    static_assert(std::is_base_of_v<NPrivate::TTrackedEventTag, T>, "expected a subclass of 'TTrackedEvent'");
    system->Send(new typename T::THandle(recipient, sender, std::move(event), flags, cookie));
}
template <typename T>
inline void SendTracked(NActors::TActorId recipient, NActors::TActorId sender,
                        THolder<T> event, ui32 flags = 0, ui64 cookie = 0)
{
    static_assert(std::is_base_of_v<NPrivate::TTrackedEventTag, T>, "expected a subclass of 'TTrackedEvent'");
    auto system = NActors::TActorContext::ActorSystem();
    SendTracked(system, recipient, sender, std::move(event), flags, cookie);
}
template <typename T>
inline void SendTracked(NActors::TActorId recipient,
                        THolder<T> event, ui32 flags = 0, ui64 cookie = 0)
{
    static_assert(std::is_base_of_v<NPrivate::TTrackedEventTag, T>, "expected a subclass of 'TTrackedEvent'");
    auto sender = NActors::TActorContext::AsActorContext().SelfID;
    SendTracked(recipient, sender, std::move(event), flags, cookie);
}

} // namespace NSolomon
