#pragma once

#include <solomon/libs/cpp/actors/fwd.h>
#include <solomon/libs/cpp/actors/events/events.h>

#include <library/cpp/actors/core/actor_bootstrapped.h>
#include <library/cpp/actors/core/event_local.h>
#include <library/cpp/actors/core/hfunc.h>
#include <library/cpp/monlib/metrics/fwd.h>

#include <util/datetime/base.h>

namespace NSolomon {

template <typename TDerived>
class TSelfPingActor: public NActors::TActorBootstrapped<TSelfPingActor<TDerived>>, private TPrivateEvents {
    enum {
        EvPing = SpaceBegin,
        End,
    };
    static_assert(End < SpaceEnd, "too many event types");

    struct TEvPing: public NActors::TEventLocal<TEvPing, EvPing> {
        explicit TEvPing(TInstant start) noexcept
            : Start{start}
        {
        }

        TInstant Start;
    };

public:
    explicit TSelfPingActor(TDuration interval) noexcept
        : Interval_{interval}
    {
    }

    void Bootstrap(const NActors::TActorContext& ctx) {
        this->Become(&TSelfPingActor::StateWork);
        this->DoPing(ctx.Now());
    }

    STFUNC(StateWork) {
        switch (ev->GetTypeRewrite()) {
            HFunc(TEvPing, Handle);
        }
    }

private:
    void DoPing(TInstant now) {
        this->Schedule(Interval_, new TEvPing{now});
    }

    void Handle(const typename TEvPing::TPtr& ev, const NActors::TActorContext& ctx) {
        const auto now = ctx.Now();
        static_cast<TDerived*>(this)->OnPing(now, ev->Get()->Start);
        DoPing(now);
    }

private:
    TDuration Interval_;
};

/**
 * Creates an actor who will periodically send itself an event and measure its delivery latency.
 * The actor will assign the resulting value to a given gauge metric.
 *
 * @param interval  event schedule delay
 * @param metric    metric to report delay latency to (in microseconds)
 *
 * @return an actor instance.
 */
std::unique_ptr<NActors::IActor> CreateLatencyActor(TDuration interval, NMonitoring::TIntGauge* metric);

/**
 * Creates an actor who will periodically get the running time of the current process.
 * The actor will assign the resulting value to a given counter metric.
 *
 * @param metric    metric to report process uptime to (in milliseconds)
 *
 * @return an actor instance.
 */
std::unique_ptr<NActors::IActor> CreateUptimeActor(NMonitoring::ICounter* metric);

} // namespace NSolomon::NFetcher
