#pragma once

#include <library/cpp/monlib/metrics/metric_registry.h>
#include <library/cpp/actors/core/mon_stats.h>

namespace NSolomon {

struct TTreadPoolMetrics {
public:
    TTreadPoolMetrics(NMonitoring::TMetricRegistry& registry, TStringBuf name)
        : Registry_(registry)
        , Name_(name)
    {
        MaxSize = registry.IntGauge({{"sensor", "actors.threads.maxSize"}, {"id", name}});
        Alive = registry.IntGauge({{"sensor", "actors.alive"}, {"id", name}});
        Registered = registry.Rate({{"sensor", "actors.registered"}, {"id", name}});
        Destroyed = registry.Rate({{"sensor", "actors.destroyed"}, {"id", name}});
        Mailboxes = registry.IntGauge({{"sensor", "actors.mailboxes"}, {"id", name}});
        CpuTimeMs = registry.Rate({{"sensor", "actors.threads.cpuTimeMs"}, {"id", name}});
        UseTimeMs = registry.Rate({{"sensor", "actors.threads.useTimeMs"}, {"id", name}});
        ParkTimeMs = registry.Rate({{"sensor", "actors.threads.parkTimeMs"}, {"id", name}});
        BlockTimeMs = registry.Rate({{"sensor", "actors.threads.blockTimeMs"}, {"id", name}});
        SentEvents = registry.Rate({{"sensor", "actors.events.sent"}, {"id", name}});
        ReceivedEvents = registry.Rate({{"sensor", "actors.events.receive"}, {"id", name}});
        DroppedEvents = registry.Rate({{"sensor", "actors.events.drop"}, {"id", name}});
        ActivationTimeMs = registry.HistogramRate({{"sensor", "actors.activationTimeMs"}, {"id", name}},
            NMonitoring::ExponentialHistogram(16, 2, 1));
        DeliveryTimeMs = registry.HistogramRate({{"sensor", "actors.events.deliveryTimeMs"}, {"id", name}},
            NMonitoring::ExponentialHistogram(16, 2, 1));
        ProcessingTimeMs = registry.HistogramRate({{"sensor", "actors.events.processingTimeMs"}, {"id", name}},
            NMonitoring::ExponentialHistogram(16, 2, 1));
        MailboxEmptyActivation = registry.Rate({{"sensor", "actors.mailbox.emptyActivation"}, {"id", name}});
        MailboxPushedOutByTime = registry.Rate({{"sensor", "actors.mailbox.pushOut.byTime"}, {"id", name}});
        MailboxPushedOutByEventCount = registry.Rate({{"sensor", "actors.mailbox.pushOut.byEventCount"}, {"id", name}});
    }

    void Update(NActors::TExecutorThreadStats& stats) {
        Alive->Set(stats.PoolActorRegistrations - stats.PoolDestroyedActors);
        Mailboxes->Set(stats.PoolAllocatedMailboxes);
        Add(Registered, stats.PoolActorRegistrations);
        Add(Destroyed, stats.PoolDestroyedActors);

        Add(SentEvents, stats.SentEvents);
        Add(ReceivedEvents, stats.ReceivedEvents);
        Add(DroppedEvents, stats.NonDeliveredEvents);

        Add(CpuTimeMs, stats.CpuNs / 1000'000);
        AddMs(UseTimeMs, stats.ElapsedTicks);
        AddMs(ParkTimeMs, stats.ParkedTicks);
        AddMs(BlockTimeMs, stats.BlockedTicks);

        Add(ActivationTimeMs, Delta(stats.ActivationTimeHistogram, PrevActivationTime_));
        PrevActivationTime_ = stats.ActivationTimeHistogram;

        Add(DeliveryTimeMs, Delta(stats.EventDeliveryTimeHistogram, PrevDeliveryTime_));
        PrevDeliveryTime_ = stats.EventDeliveryTimeHistogram;

        Add(ProcessingTimeMs, Delta(stats.EventProcessingCountHistogram, PrevProcessTime_));
        PrevProcessTime_ = stats.EventProcessingCountHistogram;

        Add(MailboxEmptyActivation, stats.EmptyMailboxActivation);
        Add(MailboxPushedOutByTime, stats.MailboxPushedOutByTime);
        Add(MailboxPushedOutByEventCount, stats.MailboxPushedOutByEventCount);

        Set(ActorsAliveByActivity, stats.ActorsAliveByActivity, [&](size_t activity) {
            return Registry_.IntGauge({
                {"sensor", "actors.alive"},
                {"id", Name_},
                {"activity", TString(NActors::GetActivityTypeName(activity))}});
        });

        AddMs(ElapsedTicksByActivity, stats.ElapsedTicksByActivity, [&](size_t activity) {
            return Registry_.Rate({
                {"sensor", "actors.threads.useTimeMs"},
                {"id", Name_},
                {"activity", TString(NActors::GetActivityTypeName(activity))}});
        });
        Add(ReceivedEventsByActivity, stats.ReceivedEventsByActivity, [&](size_t activity) {
            return Registry_.Rate({
                {"sensor", "actors.events.receive"},
                {"id", Name_},
                {"activity", TString(NActors::GetActivityTypeName(activity))}});
        });
        Add(ScheduledEventsByActivity, stats.ScheduledEventsByActivity, [&](size_t activity) {
            return Registry_.Rate({
                {"sensor", "actors.events.scheduled"},
                {"id", Name_},
                {"activity", TString(NActors::GetActivityTypeName(activity))}});
        });
    }

    static NActors::TLogHistogram Delta(const NActors::TLogHistogram& now, const NActors::TLogHistogram& prev) {
        NActors::TLogHistogram result;
        for (size_t i = 0; i < Y_ARRAY_SIZE(result.Buckets); i++) {
            result.Buckets[i] = now.Buckets[i] - prev.Buckets[i];
        }
        return result;
    }

    static void Add(NMonitoring::THistogram* target, const NActors::TLogHistogram& delta) {
        for (size_t i = 0; i < Y_ARRAY_SIZE(delta.Buckets); i++) {
            ui64 count = delta.Buckets[i];
            if (count == 0) {
                continue;
            }
            auto usec = 1ull<<i;
            target->Record(usec / 1000, count);
        }
    }

    static void Add(NMonitoring::TRate* target, ui64 monoCounter) {
        target->Add(monoCounter - target->Get());
    }

    static void AddMs(NMonitoring::TRate* target, NHPTimer::STime time) {
        ui64 ms = ::NHPTimer::GetSeconds(time) * 1000;
        Add(target, ms);
    }

    template <typename T, typename Function>
    void Add(TVector<NMonitoring::TRate*>& target, const TVector<T>& rates, Function&& makeRate) {
        FillMetrics(target, rates, makeRate);
        for (size_t i = 0; i < target.size(); ++i) {
            if (target[i]) {
                Add(target[i], rates[i]);
            }
        }
    }

    template <typename T, typename Function>
    void AddMs(TVector<NMonitoring::TRate*>& target, const TVector<T>& rates, Function&& makeRate) {
        FillMetrics(target, rates, makeRate);
        for (size_t i = 0; i < target.size(); ++i) {
            if (target[i]) {
                AddMs(target[i], rates[i]);
            }
        }
    }

    template <typename TMetric, typename T, typename Function>
    void FillMetrics(TVector<TMetric*>& target, const TVector<T>& values, Function&& makeMetric) {
        if (target.size() < values.size()) {
            size_t oldSize = target.size();
            target.resize(values.size());
            for (size_t i = oldSize; i < target.size(); ++i) {
                if (values[i] > 0) {
                    target[i] = makeMetric(i);
                }
            }
        }
    }

    template <typename T, typename Function>
    void Set(TVector<NMonitoring::TIntGauge*>& target, const TVector<T>& gauges, Function&& makeGauge) {
        FillMetrics(target, gauges, makeGauge);
        for (size_t i = 0; i < target.size(); ++i) {
            if (target[i]) {
                target[i]->Set(gauges[i]);
            }
        }
    }

public:
    NMonitoring::TIntGauge* MaxSize;
    NMonitoring::TIntGauge* Alive;
    NMonitoring::TIntGauge* Mailboxes;
    NMonitoring::TRate* Registered;
    NMonitoring::TRate* Destroyed;
    NMonitoring::TRate* SentEvents;
    NMonitoring::TRate* ReceivedEvents;
    NMonitoring::TRate* DroppedEvents;
    NMonitoring::TRate* CpuTimeMs;
    NMonitoring::TRate* UseTimeMs;
    NMonitoring::TRate* ParkTimeMs;
    NMonitoring::TRate* BlockTimeMs;
    NMonitoring::THistogram* ActivationTimeMs;
    NMonitoring::THistogram* DeliveryTimeMs;
    NMonitoring::THistogram* ProcessingTimeMs;
    NMonitoring::TRate* MailboxEmptyActivation;
    NMonitoring::TRate* MailboxPushedOutByTime;
    NMonitoring::TRate* MailboxPushedOutByEventCount;
    TVector<NMonitoring::TRate*> ElapsedTicksByActivity;
    TVector<NMonitoring::TRate*> ReceivedEventsByActivity;
    TVector<NMonitoring::TIntGauge*> ActorsAliveByActivity;
    TVector<NMonitoring::TRate*> ScheduledEventsByActivity;

private:
    NActors::TLogHistogram PrevActivationTime_;
    NActors::TLogHistogram PrevDeliveryTime_;
    NActors::TLogHistogram PrevProcessTime_;

    NMonitoring::TMetricRegistry& Registry_;
    TString Name_;
};

} // namespace NSolomon
