#include "threadpool_metrics_collector.h"
#include "threadpool_metrics.h"

#include <library/cpp/actors/core/actor.h>
#include <library/cpp/actors/core/actor_bootstrapped.h>
#include <library/cpp/actors/core/hfunc.h>

using namespace NActors;
using namespace NMonitoring;

namespace NSolomon {
namespace {
class TActorMetricsCollector : public TActorBootstrapped<TActorMetricsCollector> {
public:
    TActorMetricsCollector(TDuration interval, std::vector<TTreadPoolMetrics> metrics)
        : Interval_(interval)
        , ThreadPoolsMetrics_(std::move(metrics))
    {
    }

    void Bootstrap(const TActorContext& ctx) {
        Y_UNUSED(ctx);
        Become(&TActorMetricsCollector::StateNormal);
        Schedule(Interval_, new TEvents::TEvWakeup());
    }

    STFUNC(StateNormal) {
        switch (ev->GetTypeRewrite()) {
            hFunc(TEvents::TEvPoison, OnPoison);
            CFunc(TEvents::TSystem::Wakeup, OnWakeup);
        }
    }

    void OnWakeup(const TActorContext &ctx) {
        Schedule(Interval_, new TEvents::TEvWakeup());
        for (size_t poolId = 0; poolId < ThreadPoolsMetrics_.size(); poolId++) {
            TVector<TExecutorThreadStats> stats;
            TExecutorPoolStats poolStats;
            ctx.ExecutorThread.ActorSystem->GetPoolStats(poolId, poolStats, stats);
            if (stats.empty()) {
                continue;
            }

            auto& metrics = ThreadPoolsMetrics_[poolId];
            metrics.MaxSize->Set(stats.size() - 1);
            auto& aggrStats = stats[0];
            for (size_t threadIdx = 1; threadIdx < stats.size(); threadIdx++) {
                aggrStats.Aggregate(stats[threadIdx]);
            }

            metrics.Update(aggrStats);
        }
    }

    void OnPoison(TEvents::TEvPoison::TPtr& ev) {
        Send(ev->Sender, new TEvents::TEvPoisonTaken{});
        PassAway();
    }

private:
    const TDuration Interval_;
    std::vector<TTreadPoolMetrics> ThreadPoolsMetrics_;
};
}

std::unique_ptr<NActors::IActor> CreateThreadPoolMetricCollector(TDuration interval, std::vector<TTreadPoolMetrics> metrics) {
    return std::make_unique<TActorMetricsCollector>(interval, std::move(metrics));
}
} // namespace NSolomon
