#include "async_poison.h"

#include <library/cpp/actors/core/actor_bootstrapped.h>
#include <library/cpp/actors/core/actorsystem.h>
#include <library/cpp/actors/core/hfunc.h>
#include <library/cpp/threading/future/future.h>

using namespace NActors;

namespace NSolomon {
namespace {

class TAsyncPoisonerActor: public TActorBootstrapped<TAsyncPoisonerActor> {
public:
    TAsyncPoisonerActor(std::set<TActorId> toPoison, TDuration timeout, NThreading::TPromise<void> promise) noexcept
        : ToPoison_{std::move(toPoison)}
        , Timeout_{timeout}
        , Promise_{std::move(promise)}
    {
    }

    void Bootstrap() {
        Become(&TThis::Poisoning);

        if (Timeout_) {
            Schedule(Timeout_, new TEvents::TEvWakeup);
        }

        for (TActorId actorId: ToPoison_) {
            Send(actorId, new TEvents::TEvPoison);
        }
    }

    STATEFN(Poisoning) {
        switch (ev->GetTypeRewrite()) {
            hFunc(TEvents::TEvPoisonTaken, OnPoisonTaken)
            sFunc(TEvents::TEvWakeup, OnTimeout)
        }
    }

    void OnPoisonTaken(const TEvents::TEvPoisonTaken::TPtr& ev) {
        if (ToPoison_.erase(ev->Sender) != 0 && ToPoison_.empty()) {
            Promise_.SetValue();
            PassAway();
        }
    }

    void OnTimeout() {
        Promise_.SetValue();
        PassAway();
    }

private:
    std::set<TActorId> ToPoison_;
    TDuration Timeout_;
    NThreading::TPromise<void> Promise_;
};

} // namespace

NThreading::TFuture<void> AsyncPoison(TActorSystem* actorSystem, std::set<TActorId> toPoison, TDuration timeout) {
    // drop an empty actor ID to avoid waiting infinite time
    toPoison.erase(TActorId{});
    if (toPoison.empty()) {
        return NThreading::MakeFuture();
    }

    auto promise = NThreading::NewPromise();
    auto future = promise.GetFuture();
    actorSystem->Register(new TAsyncPoisonerActor{std::move(toPoison), timeout, std::move(promise)});
    return future;
}

} // namespace NSolomon
