#include "poisoner.h"

#include <library/cpp/actors/core/actor.h>
#include <library/cpp/actors/core/hfunc.h>

using namespace NActors;

namespace NSolomon {
namespace {

class TPoisonerActor: public TActor<TPoisonerActor> {
public:
    TPoisonerActor(std::set<TActorId> toPoison, TDuration timeout) noexcept
        : TActor<TPoisonerActor>{&TThis::AwaitingPoison}
        , ToPoison_{std::move(toPoison)}
        , Timeout_{timeout}
    {
    }

    STATEFN(AwaitingPoison) {
        switch (ev->GetTypeRewrite()) {
            hFunc(TEvents::TEvPoison, OnPoison)
        }
    }

    STATEFN(Poisoning) {
        switch (ev->GetTypeRewrite()) {
            hFunc(TEvents::TEvPoisonTaken, OnPoisonTaken)
            sFunc(TEvents::TEvWakeup, OnTimeout)
        }
    }

    void OnPoison(const TEvents::TEvPoison::TPtr& ev) {
        Become(&TThis::Poisoning);
        ReplyTo_ = ev->Sender;

        if (Timeout_) {
            Schedule(Timeout_, new TEvents::TEvWakeup);
        }

        for (TActorId actorId: ToPoison_) {
            Send(actorId, new TEvents::TEvPoison);
        }
    }

    void OnPoisonTaken(const TEvents::TEvPoisonTaken::TPtr& ev) {
        if (ToPoison_.erase(ev->Sender) != 0 && ToPoison_.empty()) {
            Send(ReplyTo_, new TEvents::TEvPoisonTaken);
            PassAway();
        }
    }

    void OnTimeout() {
        Send(ReplyTo_, new TEvents::TEvPoisonTaken);
        PassAway();
    }

private:
    TActorId ReplyTo_;
    std::set<TActorId> ToPoison_;
    TDuration Timeout_;
};

} // namespace

std::unique_ptr<IActor> CreatePoisoner(std::set<TActorId> toPoison, TDuration timeout) {
    return std::make_unique<TPoisonerActor>(std::move(toPoison), timeout);
}

void PoisonAll(const NActors::TEvents::TEvPoison::TPtr& event, std::set<NActors::TActorId> toPoison, TDuration timeout) {
    auto& ctx = NActors::TActivationContext::AsActorContext();
    auto poisoner = ctx.Register(new TPoisonerActor{std::move(toPoison), timeout});
    ctx.Send(event->Forward(poisoner));
}

} // namespace NSolomon
