#include "test_actor.h"

#include <solomon/libs/cpp/actors/poison/async_poison.h>
#include <solomon/libs/cpp/actors/test_runtime/actor_runtime.h>

#include <library/cpp/testing/gtest/gtest.h>

using namespace NActors;
using namespace NSolomon;

class TAsyncPoisonTest: public ::testing::Test {
protected:
    void SetUp() override {
        Runtime_ = TTestActorRuntime::CreateInited();
    }

    TActorId CreateTestActor(TDuration delay = {}) {
        return Runtime_->Register(new TTestActor{&NumOfPoisoned_, delay});
    }

protected:
    THolder<TTestActorRuntime> Runtime_;
    std::atomic<size_t> NumOfPoisoned_{0};
};

TEST_F(TAsyncPoisonTest, Simple) {
    auto one = CreateTestActor();
    auto two = CreateTestActor();

    auto future = AsyncPoison(Runtime_->SingleSys(), std::set{one, two});
    Runtime_->WaitForBootstrap();

    ASSERT_TRUE(future.HasValue());
    ASSERT_EQ(NumOfPoisoned_.load(), 2u);

    ASSERT_FALSE(Runtime_->FindActor(one));
    ASSERT_FALSE(Runtime_->FindActor(two));
}

TEST_F(TAsyncPoisonTest, Timeout) {
    auto one = CreateTestActor(TDuration::Seconds(5));
    auto two = CreateTestActor(TDuration::Seconds(20));

    auto future = AsyncPoison(Runtime_->SingleSys(), std::set{one, two}, TDuration::Seconds(10));
    Runtime_->DispatchEvents({}, TDuration::Seconds(5));

    ASSERT_TRUE(future.HasValue());
    ASSERT_EQ(NumOfPoisoned_.load(), 1u);

    // first actor is dead
    ASSERT_FALSE(Runtime_->FindActor(one));

    // second actor still alive
    ASSERT_TRUE(Runtime_->FindActor(two));
}

TEST_F(TAsyncPoisonTest, EmptyId) {
    auto future = AsyncPoison(Runtime_->SingleSys(), std::set{TActorId{}}, TDuration::Seconds(5));
    Runtime_->DispatchEvents({}, TDuration::Seconds(5));

    ASSERT_TRUE(future.HasValue());
    ASSERT_EQ(NumOfPoisoned_.load(), 0u);
}

TEST_F(TAsyncPoisonTest, EmptySet) {
    auto future = AsyncPoison(Runtime_->SingleSys(), {}, TDuration::Seconds(5));
    Runtime_->DispatchEvents({}, TDuration::Seconds(5));

    ASSERT_TRUE(future.HasValue());
    ASSERT_EQ(NumOfPoisoned_.load(), 0u);
}
