#include "test_actor.h"

#include <solomon/libs/cpp/actors/poison/poisoner.h>
#include <solomon/libs/cpp/actors/test_runtime/actor_runtime.h>

#include <library/cpp/testing/gtest/gtest.h>

using namespace NActors;
using namespace NSolomon;

class TPoisonerTest: public ::testing::Test {
protected:
    void SetUp() override {
        Runtime_ = TTestActorRuntime::CreateInited();
        Edge_ = Runtime_->AllocateEdgeActor();
    }

    TActorId CreateTestActor(TDuration delay = {}) {
        return Runtime_->Register(new TTestActor{&NumOfPoisoned_, delay});
    }

    template <typename... Args>
    TActorId CreatePoisonerActor(Args... agrs) {
        return Runtime_->Register(CreatePoisoner(std::forward<Args>(agrs)...).release());
    }

    void Poison(TActorId recipient) {
        Runtime_->Send(recipient, Edge_, MakeHolder<TEvents::TEvPoison>());
        Runtime_->GrabEdgeEvent<TEvents::TEvPoisonTaken>(Edge_);
    }

protected:
    THolder<TTestActorRuntime> Runtime_;
    TActorId Edge_;
    std::atomic<size_t> NumOfPoisoned_{0};
};

TEST_F(TPoisonerTest, Simple) {
    auto one = CreateTestActor();
    auto two = CreateTestActor();
    auto poisoner = CreatePoisonerActor(std::set{one, two});

    Poison(poisoner);

    ASSERT_EQ(NumOfPoisoned_.load(), 2u);
    ASSERT_FALSE(Runtime_->FindActor(one));
    ASSERT_FALSE(Runtime_->FindActor(two));
    ASSERT_FALSE(Runtime_->FindActor(poisoner));
}

TEST_F(TPoisonerTest, Timeout) {
    auto one = CreateTestActor(TDuration::Seconds(5));
    auto two = CreateTestActor(TDuration::Seconds(20));
    auto poisoner = CreatePoisonerActor(std::set{one, two}, TDuration::Seconds(10));

    Poison(poisoner);
    ASSERT_EQ(NumOfPoisoned_.load(), 1u);

    // first actor and poisoner are dead
    ASSERT_FALSE(Runtime_->FindActor(one));
    ASSERT_FALSE(Runtime_->FindActor(poisoner));

    // second actor still alive
    ASSERT_TRUE(Runtime_->FindActor(two));
}

TEST_F(TPoisonerTest, Cascade) {
    auto shardA1 = CreateTestActor();
    auto shardA2 = CreateTestActor();
    auto a = CreatePoisonerActor(std::set{shardA1, shardA2});

    auto shardB1 = CreateTestActor();
    auto shardB2 = CreateTestActor();
    auto b = CreatePoisonerActor(std::set{shardB1, shardB2});

    auto root = CreatePoisonerActor(std::set{a, b});
    Poison(root);

    ASSERT_EQ(NumOfPoisoned_.load(), 4u);

    ASSERT_FALSE(Runtime_->FindActor(shardA1));
    ASSERT_FALSE(Runtime_->FindActor(shardA2));
    ASSERT_FALSE(Runtime_->FindActor(a));

    ASSERT_FALSE(Runtime_->FindActor(shardB1));
    ASSERT_FALSE(Runtime_->FindActor(shardB2));
    ASSERT_FALSE(Runtime_->FindActor(b));

    ASSERT_FALSE(Runtime_->FindActor(root));
}
