#include "actor_runtime.h"

#include <solomon/libs/cpp/actors/config/log_component.pb.h>
#include <solomon/libs/cpp/actors/events/common.h>
#include <solomon/libs/cpp/actors/poison/async_poison.h>
#include <solomon/libs/cpp/actors/interconnect/init.h>

#include <library/cpp/actors/interconnect/poller_tcp.h>
#include <library/cpp/actors/core/actorsystem.h>
#include <library/cpp/actors/core/executor_pool_basic.h>
#include <library/cpp/actors/core/scheduler_basic.h>
#include <library/cpp/actors/core/log.h>

#include <utility>

namespace NSolomon {
namespace {

using namespace NActors;
using namespace NMonitoring;

TActorId MakeLoggerActorID() {
    return TActorId(0, TStringBuf("Logger"));
}

IExecutorPool* CreateExecutorPool(ui32 id, const TExecutorConfig& config, ui32 maxActivityType) {
    constexpr static ui32 DEFAULT_THREAD_COUNT = 4;

    ui32 threads = (config.threads() != 0) ? config.threads() : DEFAULT_THREAD_COUNT;
    return new TBasicExecutorPool(
            id,
            threads,
            20,
            config.name(),
            nullptr,
            TBasicExecutorPool::DEFAULT_TIME_PER_MAILBOX,
            TBasicExecutorPool::DEFAULT_EVENTS_PER_MAILBOX,
            0,
            maxActivityType);
}

TIntrusivePtr<NLog::TSettings> CreateLogSettings(NLog::EPriority priority) {
    const auto loggerId = MakeLoggerActorID();
    auto settings = MakeIntrusive<NLog::TSettings>(loggerId, ELogComponent::Logger, priority);
    settings->UseLocalTimestamps = true;
    settings->Append(ELogComponent_MIN, ELogComponent_MAX, ELogComponent_Name);
    return settings;
}

TAutoPtr<TLogBackend> CreateBackend(const TLogConfig& config) {
    switch (config.GetType()) {
    case TLogConfig::STDERR:
        return CreateStderrBackend();
    case TLogConfig::FILE: {
        const auto& fileName = config.GetFileName();
        Y_ENSURE(!fileName.Empty(), "File name must be specified for the FILE log type");
        return CreateFileBackend(fileName);
    }
    default:
        ythrow yexception() << "Unknown logger type: " << TLogConfig::EType_Name(config.GetType());
    }
}

} // namespace

void TActorRuntime::InitInterconnect(
    TActorSystemSetup& setup,
    const TActorSystemConfig& actorSystemConfig,
    std::shared_ptr<NMonitoring::TMetricRegistry> metrics)
{
    if (!actorSystemConfig.HasInterconnectConfig()) {
        return;
    }

    auto&& config = actorSystemConfig.GetInterconnectConfig();
    NSolomon::InitInterconnect(setup, config, std::move(metrics));
}

TActorRuntime::TPtr TActorRuntime::Create(
    const TActorSystemConfig& config,
    const std::shared_ptr<NMonitoring::TMetricRegistry>& metrics,
    void* appData)
{
    Y_ENSURE(config.ExecutorsSize() > 0, "At least one executor must be specified");

    auto setup = MakeHolder<TActorSystemSetup>();
    setup->ExecutorsCount = config.ExecutorsSize() + 1; // +1 for logger executor
    setup->Executors.Reset(new TAutoPtr<IExecutorPool>[setup->ExecutorsCount]);
    setup->MaxActivityType = GetActivityTypeCount();

    auto runtime = MakeHolder<TActorRuntime>();
    ui32 poolId = 0;
    for (auto i = 0u; i < config.ExecutorsSize(); ++i) {
        auto&& executorConfig = config.GetExecutors(i);
        setup->Executors[poolId].Reset(CreateExecutorPool(poolId, executorConfig, setup->MaxActivityType));
        runtime->AddExecutorPool(executorConfig.GetName(), poolId);
        ++poolId;
    }

    // logger must use separate executor to avoid deadlocks
    const ui32 loggerPoolId = poolId;
    {
        TString loggerPoolName{"logger"};
        setup->Executors[loggerPoolId].Reset(new TBasicExecutorPool(
                loggerPoolId,
                1,
                20,
                loggerPoolName,
                nullptr,
                TBasicExecutorPool::DEFAULT_TIME_PER_MAILBOX,
                TBasicExecutorPool::DEFAULT_EVENTS_PER_MAILBOX,
                0,
                setup->MaxActivityType));
        runtime->AddExecutorPool(std::move(loggerPoolName), loggerPoolId);
    }

    setup->Scheduler.Reset(new TBasicSchedulerThread);

    TIntrusivePtr<NLog::TSettings> settings;
    if (!config.HasLogConfig()) {
        Cerr << "log config not found -- using default settings" << Endl;
        settings = CreateLogSettings(NLog::PRI_ERROR);
    } else {
        auto&& logConf = config.GetLogConfig();
        NLog::EPriority defaultPriority{NLog::PRI_ERROR};
        if (auto confPriority = logConf.GetDefaultLevel()) {
            defaultPriority = static_cast<NLog::EPriority>(confPriority);
        }

        settings = CreateLogSettings(defaultPriority);

        if (logConf.GetOnOverflow() == TLogConfig_EOnOverflow_DROP) {
            settings->SetAllowDrop(true);
        } else if (logConf.GetOnOverflow() == TLogConfig_EOnOverflow_BLOCK) {
            settings->SetAllowDrop(false);
        }

        if (logConf.GetDefaultSamplingLevel()) {
            settings->DefSamplingPriority = static_cast<NLog::EPriority>(logConf.GetDefaultSamplingLevel());
        }

        if (logConf.GetDefaultSamplingRate()) {
            settings->DefSamplingRate = logConf.GetDefaultSamplingRate();
        }

        if (logConf.GetFormat().empty() || logConf.GetFormat() == TStringBuf("full")) {
            settings->Format = NLog::TSettings::PLAIN_FULL_FORMAT;
        } else if (logConf.GetFormat() == TStringBuf("short")) {
            settings->Format = NLog::TSettings::PLAIN_SHORT_FORMAT;
        } else if (logConf.GetFormat() == TStringBuf("json")) {
            settings->Format = NLog::TSettings::JSON_FORMAT;
        }

        for (auto&& entry: logConf.GetEntry()) {
            ELogComponent component;
            Y_ENSURE(ELogComponent_Parse(entry.GetComponent(), &component),
                     "unknown log component: " << entry.GetComponent());

            auto priority = entry.GetLevel() ? static_cast<NLog::EPriority>(entry.GetLevel()) : defaultPriority;
            auto samplingLevel = static_cast<NLog::EPriority>(entry.GetSamplingLevel());

            TString explanation;
            Y_VERIFY(settings->SetLevel(priority, component, explanation) == 0);
            Y_VERIFY(settings->SetSamplingLevel(samplingLevel, component, explanation) == 0);
            Y_VERIFY(settings->SetSamplingRate(entry.GetSamplingRate(), component, explanation) == 0);
        }
    }

    auto logBackend = CreateBackend(config.GetLogConfig());
    runtime->Logger_ = logBackend.Get();

    TActorSetupCmd loggerActorCmd;
    loggerActorCmd.Actor = new TLoggerActor(settings, logBackend, metrics);

    loggerActorCmd.PoolId = loggerPoolId;
    setup->LocalServices.emplace_back(MakeLoggerActorID(), loggerActorCmd);

    runtime->InitInterconnect(*setup, config, metrics);
    runtime->ActorSystem_ = MakeHolder<TActorSystem>(setup, appData, settings);

    return runtime;
}

void TActorRuntime::Start() {
    ActorSystem_->Start();
}

TActorRuntime::~TActorRuntime() {
    Stop();
}

void TActorRuntime::Stop() {
    if (ActorSystem_) {
        ActorSystem_->Stop();
    }

    if (ActorSystem_) {
        ActorSystem_->Cleanup();
        ActorSystem_.Reset();
    }
}

TActorId TActorRuntime::Register(
    IActor* actor,
    TMailboxType::EType mailboxType,
    ui32 executorPool,
    ui64 revolvingCounter,
    const NActors::TActorId& parentId)
{
    Y_VERIFY(ActorSystem_);
    return ActorSystem_->Register(actor, mailboxType, executorPool, revolvingCounter, parentId);
}

void TActorRuntime::RegisterLocalService(const TActorId& serviceId, const TActorId& actorId) {
    Y_VERIFY(ActorSystem_);
    ActorSystem_->RegisterLocalService(serviceId, actorId);
}

NThreading::TFuture<void> TActorRuntime::AsyncPoison(NActors::TActorId actorId) {
    auto poison = std::make_unique<TCommonEvents::TAsyncPoison>();
    auto future = poison->Future();
    ActorSystem_->Send(actorId, poison.release());
    return future;
}

NThreading::TFuture<void> TActorRuntime::AsyncPoison(std::set<NActors::TActorId> toPoison, TDuration timeout) {
    return NSolomon::AsyncPoison(ActorSystem_.Get(), std::move(toPoison), timeout);
}

ui32 TActorRuntime::FindExecutorByName(const TString& name) const {
    auto it = ExecutorPools_.find(name);
    Y_ENSURE(it != ExecutorPools_.end(), "cannot find executor by name: \'" << name << '\'');
    return it->second;
}

void TActorRuntime::AddExecutorPool(TString name, ui32 id) {
    auto [nameIt, nameUniq] = ExecutorPools_.emplace(std::move(name), id);
    Y_ENSURE(nameUniq, "used non unique executor name: \'" << nameIt->first << '\'');

    auto [_, idUniq] = ExecutorPoolsReverseIdx_.emplace(id, nameIt->first);
    Y_ENSURE(idUniq, "used non unique executor id: " << id);
}

TStringBuf TActorRuntime::ExecutorName(ui32 id) {
    auto it = ExecutorPoolsReverseIdx_.find(id);
    return it == ExecutorPoolsReverseIdx_.end() ? TStringBuf{} : it->second;
}

ui32 TActorRuntime::ExecutorsSize() {
    return ExecutorPools_.size();
}

void TActorRuntime::ReopenLog() {
    Logger_->ReopenLog();
}

void TActorRuntime::EmergencyLog(TStringBuf msg) {
    Logger_->WriteData(TLogRecord{ELogPriority::TLOG_EMERG, msg.Data(), msg.Size()});
}

} // namespace NSolomon::NFetcher
