#include "scheduler.h"

#include <solomon/libs/cpp/logging/logging.h>
#include <solomon/libs/cpp/threading/timer/timer_wheel.h>

#include <library/cpp/actors/core/actor_bootstrapped.h>
#include <library/cpp/actors/core/hfunc.h>
#include <library/cpp/containers/absl_flat_hash/flat_hash_map.h>

#include <util/digest/multi.h>

namespace NSolomon {
namespace {

struct TEventId {
    NActors::TActorId Sender;
    ui64 EventId;

    bool operator==(const TEventId& other) const {
        return Sender == other.Sender && EventId == other.EventId;
    }
};

} // namespace
} // namespace NSolomon

template <>
struct std::hash<NSolomon::TEventId> {
    size_t operator()(const NSolomon::TEventId& id) const noexcept {
        return MultiHash(id.Sender, id.EventId);
    }
};

namespace NSolomon {
namespace {

using namespace NActors;
using namespace NSolomon::NTimer;

using TIdToEvent = absl::flat_hash_map<TEventId, std::unique_ptr<class TEvent>>;

class TEvent: public ITimerEvent {
public:
    TEvent(
            TActorSystem* actorSystem,
            TActorId replyTo,
            TActorId selfId,
            TEventId id,
            std::unique_ptr<IEventBase> replyEvent,
            ui64 cookie,
            TIdToEvent& idToEvent)
        : ActorSystem_{actorSystem}
        , ReplyTo_{replyTo}
        , SelfId_{selfId}
        , Id_{id}
        , ReplyEvent_{std::move(replyEvent)}
        , Cookie_{cookie}
        , IdToEvent_{idToEvent}
    {
    }

private:
    void Execute() override {
        ActorSystem_->Send(new IEventHandle{ReplyTo_, SelfId_, ReplyEvent_.release(), 0, Cookie_});
        IdToEvent_.erase(Id_);
    }

private:
    TActorSystem* ActorSystem_;
    TActorId ReplyTo_;
    TActorId SelfId_;
    TEventId Id_;
    std::unique_ptr<IEventBase> ReplyEvent_;
    ui64 Cookie_;
    TIdToEvent& IdToEvent_;
};

class TScheduler: public TActorBootstrapped<TScheduler> {
public:
    TScheduler(TInstant now, TDuration tickDuration)
        : Timer_{now, tickDuration}
        , TickDuration_{tickDuration}
    {
    }

    void Bootstrap() {
        Become(&TThis::StateFn);
        Schedule(TickDuration_, new TEvents::TEvWakeup);
    }

    STATEFN(StateFn) {
        switch (ev->GetTypeRewrite()) {
            hFunc(TSchedulerEvents::TScheduleAfter, OnScheduleAfter);
            hFunc(TSchedulerEvents::TScheduleAt, OnScheduleAt);
            hFunc(TSchedulerEvents::TCancel, OnCancel);

            sFunc(TEvents::TEvWakeup, AdvanceTime);
            hFunc(TEvents::TEvPoison, OnPoison);
        }
    }

private:
    void OnScheduleAfter(const TSchedulerEvents::TScheduleAfter::TPtr& evPtr) {
        auto& ev = *evPtr->Get();
        auto replyTo = ev.ReplyTo ? ev.ReplyTo : evPtr->Sender;

        if (Y_UNLIKELY(ev.Delay < Timer_.TickDuration())) {
            MON_WARN(Scheduler, "event delay (" << ev.Delay << ") < tick duration (" << Timer_.TickDuration() << ')');
            Send(replyTo, ev.ReplyEvent.release(), 0, evPtr->Cookie);
            return;
        }

        auto eventId = TEventId{
            .Sender = evPtr->Sender,
            .EventId = ev.Id,
        };

        auto reqEventPtr = std::make_unique<TEvent>(
                TActivationContext::ActorSystem(),
                replyTo,
                SelfId(),
                eventId,
                std::move(ev.ReplyEvent),
                evPtr->Cookie,
                IdToEvent_);

        auto* reqEv = reqEventPtr.get();
        IdToEvent_[eventId] = std::move(reqEventPtr);
        Timer_.ScheduleAfter(reqEv, ev.Delay);
    }

    void OnScheduleAt(const TSchedulerEvents::TScheduleAt::TPtr& evPtr) {
        auto& ev = *evPtr->Get();
        auto replyTo = ev.ReplyTo ? ev.ReplyTo : evPtr->Sender;

        auto timerNow = Timer_.Now() + Timer_.TickDuration();
        if (Y_UNLIKELY(ev.Time < timerNow)) {
            MON_WARN(Scheduler, "event time (" << ev.Time << ") < timer now (" << timerNow << ')');
            Send(replyTo, ev.ReplyEvent.release(), 0, evPtr->Cookie);
            return;
        }

        auto eventId = TEventId{
            .Sender = evPtr->Sender,
            .EventId = ev.Id,
        };

        auto reqEventPtr = std::make_unique<TEvent>(
                TActivationContext::ActorSystem(),
                replyTo,
                SelfId(),
                eventId,
                std::move(ev.ReplyEvent),
                evPtr->Cookie,
                IdToEvent_);

        auto* reqEv = reqEventPtr.get();
        IdToEvent_[eventId] = std::move(reqEventPtr);
        Timer_.ScheduleAt(reqEv, ev.Time);
    }

    void OnCancel(const TSchedulerEvents::TCancel::TPtr& evPtr) {
        auto eventId = TEventId{
            .Sender = evPtr->Sender,
            .EventId = evPtr->Get()->Id,
        };

        if (auto nh = IdToEvent_.extract(eventId)) {
            nh.mapped()->Cancel();
        }
    }

    void AdvanceTime() {
        auto delta = TActivationContext::Now() - Timer_.Now();
        if (delta >= TickDuration_) {
            Timer_.Advance(delta);
        }
        Schedule(TickDuration_, new TEvents::TEvWakeup);
    }

    void OnPoison(const TEvents::TEvPoison::TPtr& ev) {
        Send(ev->Sender, new TEvents::TEvPoisonTaken);
        PassAway();
    }

private:
    TTimerWheel Timer_;
    TDuration TickDuration_;
    TIdToEvent IdToEvent_;
};

} // namespace

std::unique_ptr<NActors::IActor> CreateScheduler(TInstant now, TDuration tickDuration) {
    return std::make_unique<TScheduler>(now, tickDuration);
}

} // namespace NSolomon
