#pragma once

#include <solomon/services/dataproxy/lib/event_slots.h>

#include <solomon/libs/cpp/actors/events/events.h>
#include <solomon/libs/cpp/actors/events/slots.h>

#include <library/cpp/actors/core/actor.h>
#include <library/cpp/actors/core/event_local.h>

namespace NSolomon {

class TSchedulerEvents: private TEventSlot<EEventSpace::Libs, ELibSlot::Scheduler> {
private:
    enum {
        ScheduleAfter = SpaceBegin,
        ScheduleAt,
        Cancel,
        End,
    };
    static_assert(End < SpaceEnd, "too many event types");

public:
    /**
     * Schedules an event with a user-defined type. This event also accepts a cookie which we'll be
     * sent back to a user as a cookie in a response event. A typical use-case looks like this:
     *
     * user ------> schedule(evPtr) -----> scheduler
     * user ......... time passes ........ scheduler
     * user ......... time passes ........ scheduler
     * user <---------   evPtr   <-------- scheduler
     *
     * A cookie value passed with a reply event is a TInstant denoting at what time an event was executed
     */
    struct TScheduleAfter: public NActors::TEventLocal<TScheduleAfter, ScheduleAfter> {
        /**
         * Id of an event, generated by a user.
         */
        ui64 Id;
        /**
         * After what amount of time a user should receive the ReplyEvent back.
         */
        TDuration Delay;
        /**
         * An event which will be sent to a user after SchedulerAfter time.
         * Can be cancelled using its id. See TCancel.
         */
        std::unique_ptr<IEventBase> ReplyEvent;
        /**
         * (optional) To what actor we should send a reply event.
         * If not set, reply to the original sender.
         */
        NActors::TActorId ReplyTo;

        TScheduleAfter(ui64 id, TDuration delay, std::unique_ptr<IEventBase> replyEvent, NActors::TActorId replyTo = {}) noexcept
            : Id{id}
            , Delay{delay}
            , ReplyEvent{std::move(replyEvent)}
            , ReplyTo{replyTo}
        {
        }
    };

    struct TScheduleAt: public NActors::TEventLocal<TScheduleAt, ScheduleAt> {
        /**
         * Id of an event, generated by a user.
         */
        ui64 Id;
        /**
         * After what amount of time a user should receive the ReplyEvent back.
         */
        TInstant Time;
        /**
         * An event which will be sent to a user after SchedulerAfter time.
         * Can be cancelled using its id. See TCancel.
         */
        std::unique_ptr<IEventBase> ReplyEvent;
        /**
         * (optional) To what actor we should send a reply event.
         * If not set, reply to the original sender.
         */
        NActors::TActorId ReplyTo;

        TScheduleAt(ui64 id, TInstant time, std::unique_ptr<IEventBase> replyEvent, NActors::TActorId replyTo = {}) noexcept
            : Id{id}
            , Time{time}
            , ReplyEvent{std::move(replyEvent)}
            , ReplyTo{replyTo}
        {
        }
    };

    /**
     * Cancels a scheduled event by an id passed in a TScheduleAfter event
     */
    struct TCancel: public NActors::TEventLocal<TCancel, Cancel> {
        /**
         * Id of an event passed inside a TScheduleAfter
         */
        ui64 Id;

        explicit TCancel(ui64 id) noexcept
            : Id{id}
        {
        }
    };
};

std::unique_ptr<NActors::IActor> CreateScheduler(TInstant now, TDuration tickDuration);

} // namespace NSolomon
