#include <solomon/libs/cpp/actors/events/events.h>
#include <solomon/libs/cpp/actors/scheduler/scheduler.h>
#include <solomon/libs/cpp/actors/test_runtime/actor_runtime.h>

#include <library/cpp/actors/core/actor.h>
#include <library/cpp/testing/gtest/gtest.h>

#include <util/generic/hash.h>
#include <util/generic/hash_set.h>

using namespace NActors;
using namespace NSolomon;
using namespace std::chrono_literals;

struct TLocalEvents: NSolomon::TPrivateEvents {
    enum {
        Executed = SpaceBegin,
        Executed2,
        End
    };

    static_assert(SpaceBegin < End, "too many events");

    struct TExecuted: TEventLocal<TExecuted, Executed> {
        ui64 Id;

        explicit TExecuted(ui64 id) noexcept
            : Id{id}
        {
        }
    };

    struct TExecuted2: TEventLocal<TExecuted2, Executed2> {
        ui64 Id;

        explicit TExecuted2(ui64 id) noexcept
            : Id{id}
        {
        }
    };
};

class TSchedulerTest: public ::testing::Test {
public:
    void SetUp() override {
        ActorRuntime_ = TTestActorRuntime::CreateInited(1, false, true);
        EdgeId_ = ActorRuntime_->AllocateEdgeActor();
    }

    template <typename TEvent>
    ui64 ScheduleAfter(TActorId scheduler, TDuration delay, ui64 cookie = 0) {
        auto evId = ++ReqId_;
        auto* event = new TSchedulerEvents::TScheduleAfter{evId, delay, std::make_unique<TEvent>(evId)};
        ActorRuntime_->Send(scheduler, EdgeId_, THolder(event), 0, cookie);
        return evId;
    }

    template <typename TEvent>
    ui64 ScheduleAt(TActorId scheduler, TInstant time, ui64 cookie = 0) {
        auto evId = ++ReqId_;
        auto* event = new TSchedulerEvents::TScheduleAt{evId, time, std::make_unique<TEvent>(evId)};
        ActorRuntime_->Send(scheduler, EdgeId_, THolder(event), 0, cookie);
        return evId;
    }

protected:
    THolder<TTestActorRuntime> ActorRuntime_;
    TActorId EdgeId_;

    ui64 ReqId_{0};
};

TEST_F(TSchedulerTest, ScheduleAfter) {
    auto scheduler = ActorRuntime_->Register(CreateScheduler(ActorRuntime_->GetCurrentTime(), 1s));
    ActorRuntime_->WaitForBootstrap();

    auto reqId1 = ScheduleAfter<TLocalEvents::TExecuted>(scheduler, 5s);
    auto reqId2 = ScheduleAfter<TLocalEvents::TExecuted>(scheduler, 5s);
    auto reqId3 = ScheduleAfter<TLocalEvents::TExecuted>(scheduler, 1h);

    auto receivedAnything =
            ActorRuntime_->GrabEdgeEvent<TLocalEvents::TExecuted>(EdgeId_, 1s);
    ASSERT_FALSE(receivedAnything); // because there were no timer tick

    ActorRuntime_->AdvanceCurrentTime(5s);

    auto ex1 = ActorRuntime_->GrabEdgeEvent<TLocalEvents::TExecuted>(EdgeId_, 1s);
    auto ex2 = ActorRuntime_->GrabEdgeEvent<TLocalEvents::TExecuted>(EdgeId_, 1s);
    ASSERT_TRUE(ex1);
    ASSERT_TRUE(ex2);
    auto ex3 = ActorRuntime_->GrabEdgeEvent<TLocalEvents::TExecuted>(EdgeId_, 1s);
    ASSERT_FALSE(ex3); // the third event is scheduled for 1 hour from now which is too far

    THashSet<ui64> expectedIds{reqId1, reqId2};
    ASSERT_NE(ex1->Get()->Id, ex2->Get()->Id);
    ASSERT_TRUE(expectedIds.contains(ex1->Get()->Id));
    ASSERT_TRUE(expectedIds.contains(ex2->Get()->Id));

    {
        ActorRuntime_->AdvanceCurrentTime(5s);
        auto receivedTheEvScheduledFor1HourFromNow =
                ActorRuntime_->GrabEdgeEvent<TLocalEvents::TExecuted>(EdgeId_, 1s);
        ASSERT_FALSE(receivedTheEvScheduledFor1HourFromNow);
    }

    ActorRuntime_->AdvanceCurrentTime(1h);
    auto oneHourEv = ActorRuntime_->GrabEdgeEvent<TLocalEvents::TExecuted>(EdgeId_, 1s);
    ASSERT_TRUE(oneHourEv);
    ASSERT_EQ(oneHourEv->Get()->Id, reqId3);
}

TEST_F(TSchedulerTest, ScheduleAt) {
    TInstant now = ActorRuntime_->GetCurrentTime();
    auto scheduler = ActorRuntime_->Register(CreateScheduler(now, 1s));
    ActorRuntime_->WaitForBootstrap();

    auto reqId1 = ScheduleAt<TLocalEvents::TExecuted>(scheduler, now + 5s);
    auto reqId2 = ScheduleAt<TLocalEvents::TExecuted>(scheduler, now + 5s);
    auto reqId3 = ScheduleAt<TLocalEvents::TExecuted>(scheduler, now + 1h);

    auto receivedAnything = ActorRuntime_->GrabEdgeEvent<TLocalEvents::TExecuted>(EdgeId_, 1s);
    ASSERT_FALSE(receivedAnything); // because there were no timer tick

    ActorRuntime_->AdvanceCurrentTime(5s);

    auto ex1 = ActorRuntime_->GrabEdgeEvent<TLocalEvents::TExecuted>(EdgeId_, 1s);
    ASSERT_TRUE(ex1);

    auto ex2 = ActorRuntime_->GrabEdgeEvent<TLocalEvents::TExecuted>(EdgeId_, 1s);
    ASSERT_TRUE(ex2);

    auto ex3 = ActorRuntime_->GrabEdgeEvent<TLocalEvents::TExecuted>(EdgeId_, 1s);
    ASSERT_FALSE(ex3); // the third event is scheduled for 1 hour from now which is too far

    THashSet<ui64> expectedIds{reqId1, reqId2};
    ASSERT_NE(ex1->Get()->Id, ex2->Get()->Id);
    ASSERT_TRUE(expectedIds.contains(ex1->Get()->Id));
    ASSERT_TRUE(expectedIds.contains(ex2->Get()->Id));

    ActorRuntime_->AdvanceCurrentTime(1h);
    auto oneHourEv = ActorRuntime_->GrabEdgeEvent<TLocalEvents::TExecuted>(EdgeId_, 1s);
    ASSERT_TRUE(oneHourEv);
    ASSERT_EQ(oneHourEv->Get()->Id, reqId3);
}

TEST_F(TSchedulerTest, ScheduledAndCancelled) {
    auto scheduler = ActorRuntime_->Register(CreateScheduler(ActorRuntime_->GetCurrentTime(), 1s));
    ActorRuntime_->WaitForBootstrap();

    ScheduleAfter<TLocalEvents::TExecuted>(scheduler, 5s);
    auto id2 = ScheduleAfter<TLocalEvents::TExecuted>(scheduler, 10s);
    ActorRuntime_->AdvanceCurrentTime(5s);

    auto executed1 = ActorRuntime_->GrabEdgeEvent<TLocalEvents::TExecuted>(EdgeId_);
    auto smthElseIsExecuted = ActorRuntime_->GrabEdgeEvent<TLocalEvents::TExecuted>(EdgeId_, 1s);
    ASSERT_TRUE(executed1);
    ASSERT_FALSE(smthElseIsExecuted);

    ActorRuntime_->Send(scheduler, EdgeId_, MakeHolder<TSchedulerEvents::TCancel>(id2));

    ActorRuntime_->AdvanceCurrentTime(5s);
    auto receivedCanceledEvent =
            ActorRuntime_->GrabEdgeEvent<TLocalEvents::TExecuted>(EdgeId_, 1s);

    ASSERT_FALSE(receivedCanceledEvent);
}

TEST_F(TSchedulerTest, WhenAndType) {
    auto now = ActorRuntime_->GetCurrentTime();
    auto scheduler = ActorRuntime_->Register(CreateScheduler(now, 1s));
    ActorRuntime_->WaitForBootstrap();

    ScheduleAfter<TLocalEvents::TExecuted>(scheduler, 5s, 42);
    ScheduleAfter<TLocalEvents::TExecuted2>(scheduler, 10s, 57);

    {
        ActorRuntime_->AdvanceCurrentTime(5s);
        auto ex = ActorRuntime_->GrabEdgeEvent<TLocalEvents::TExecuted>(EdgeId_);
        ASSERT_TRUE(ex);
        ASSERT_EQ(ex->Cookie, 42u);
    }

    {
        ActorRuntime_->AdvanceCurrentTime(5s);
        auto ex = ActorRuntime_->GrabEdgeEvent<TLocalEvents::TExecuted2>(EdgeId_);
        ASSERT_TRUE(ex);
        ASSERT_EQ(ex->Cookie, 57u);
    }
}

TEST_F(TSchedulerTest, SmallDuration) {
    auto now = ActorRuntime_->GetCurrentTime();
    auto scheduler = ActorRuntime_->Register(CreateScheduler(now, 1s));
    ActorRuntime_->WaitForBootstrap();

    // 200ms is undistinguished from $now with 1s tick
    ScheduleAfter<TLocalEvents::TExecuted>(scheduler, 200ms, 37);
    auto ex = ActorRuntime_->GrabEdgeEvent<TLocalEvents::TExecuted>(EdgeId_);
    ASSERT_TRUE(ex);
    ASSERT_EQ(ex->Cookie, 37u);
}
