#include <solomon/libs/cpp/actors/events/events.h>
#include <solomon/libs/cpp/actors/test_runtime/actor_runtime.h>

#include <library/cpp/testing/gtest/gtest.h>

#include <library/cpp/actors/core/actor_bootstrapped.h>
#include <library/cpp/actors/core/event_local.h>
#include <library/cpp/actors/core/hfunc.h>

#include <solomon/libs/cpp/actors/trace/trace_actor_events.h>

using namespace NSolomon;
using namespace NActors;

class TPingActor : public TActorBootstrapped<TPingActor>, private TPrivateEvents {
    enum {
        EvPing = SpaceBegin,
        EvDone,
        End,
    };
    static_assert(End < SpaceEnd, "too many event types");

    struct TEvPing: public NActors::TEventLocal<TEvPing, EvPing> {
        TEvPing(TInstant start) : Start(start) { }
        TInstant Start;
    };

    TActorId EdgeId_;
    size_t Iteration_ = 0;
    size_t MaxIterations_;
    TDuration TotalPing_;
public:
    TPingActor(TActorId edgeId, size_t maxIterations)
        : EdgeId_(edgeId)
        , MaxIterations_(maxIterations)
    { }

    STATEFN(Normal) {
        switch (ev->GetTypeRewrite()) {
            hFunc(TEvPing, OnPing);
        }
    }

    struct TEvDone: public NActors::TEventLocal<TEvDone, EvDone> {
        TEvDone(TDuration avgPing) : AvgPing(avgPing) { }
        TDuration AvgPing;
    };

    void DoPing() {
        Schedule(TDuration::MilliSeconds(0), new TEvPing{TActivationContext::Now()});
    }

    void Bootstrap(const TActorContext&) {
        Become(&TPingActor::Normal);
        DoPing();
    }

    void OnPing(TPingActor::TEvPing::TPtr& ev) {
        TotalPing_ += TActivationContext::Now() - ev->Get()->Start;
        Iteration_++;
        if (Iteration_ < MaxIterations_) {
            DoPing();
        } else {
            Send(EdgeId_, new TEvDone{TotalPing_ / MaxIterations_});
            MON_INFO(TraceEvents, "Ping is done");
            PassAway();
        }
    }
};

class TTraceActorEventsTest: public ::testing::Test {
protected:
    void SetUp() override {
        ActorRuntime_ = TTestActorRuntime::CreateRaw(1, true);
        ActorRuntime_->SetLogBackend(CreateStderrBackend());
        ActorRuntime_->Initialize();
        ActorRuntime_->SetLogPriority(ELogComponent::TraceEvents, NLog::PRI_TRACE);
        //ActorRuntime_->SetVerbose(true);
        ActorRuntime_->WaitForBootstrap();
        EdgeId_ = ActorRuntime_->AllocateEdgeActor();
    }

    void TearDown() override {
        ActorRuntime_.Reset();
    }

    void Register(size_t maxIterations) {
        ActorRuntime_->Register(new TPingActor(EdgeId_, maxIterations));
    }

    void GrabEvent() {
        auto response = ActorRuntime_->GrabEdgeEvent<TPingActor::TEvDone>(EdgeId_);

        Cout << response->Get()->AvgPing.MicroSeconds() << " us" << Endl;
    }

private:
    THolder<TTestActorRuntime> ActorRuntime_;
    TActorId EdgeId_;
};

TEST_F(TTraceActorEventsTest, Usage) {
    Register(10);

    GrabEvent();
}
