#include "requests_executor.h"

#include <library/cpp/actors/core/actor.h>
#include <library/cpp/actors/core/hfunc.h>

using namespace NActors;

namespace NSolomon {
namespace {

// TODO(ivanzhukov): add tests
/**
 * An actor that performs and retries given requests, and sends:
 *   a given event for every completed request
 *   a given event for when all requests are completed
 */
class TRequestsExecutorActor:
        public TActorBootstrapped<TRequestsExecutorActor>,
        private TPrivateEvents
{
    enum {
        EvResponse = SpaceBegin,
        EvPerformRequest,
        End
    };

    static_assert(End < SpaceEnd, "too many event types");

    struct TEvResponse: NActors::TEventLocal<TEvResponse, EvResponse> {
        IRequestCtxPtr ReqCtx;

        explicit TEvResponse(IRequestCtxPtr reqCtx)
            : ReqCtx{std::move(reqCtx)}
        {}
    };

    struct TEvPerformRequest: NActors::TEventLocal<TEvPerformRequest, EvPerformRequest> {
        IRequestCtxPtr ReqCtx;

        explicit TEvPerformRequest(IRequestCtxPtr reqCtx)
            : ReqCtx{std::move(reqCtx)}
        {}
    };

public:
    TRequestsExecutorActor(
            TRequestsExecutorOptions options,
            TVector<IRequestCtxPtr> reqCtxs,
            TActorId replyTo)
        : Options_{std::move(options)}
        , RequestContexts_{std::move(reqCtxs)}
        , ReplyTo_{replyTo}
        , RequestsCountdown_{RequestContexts_.size()}
    {
    }

    void Bootstrap() {
        Become(&TThis::Main);

        Start();
    }

// Event handlers
private:
    STATEFN(Main) {
        switch (ev->GetTypeRewrite()) {
            hFunc(TEvPerformRequest, OnPerformRequest);
            hFunc(TEvResponse, OnResponse);
        }
    }

    void OnResponse(TEvResponse::TPtr& ev) {
        auto reqCtxPtr = ev->Get()->ReqCtx;

        // TODO(ivanzhukov): use a timeout value
        bool shouldRetry = (reqCtxPtr->TimesPerformed() < Options_.MaxTries) && reqCtxPtr->ShouldRetry();
        if (shouldRetry) {
            if (Options_.Backoff) {
                Schedule(Options_.Backoff, new TEvPerformRequest{std::move(reqCtxPtr)});
            } else {
                PerformRequest(std::move(reqCtxPtr));
            }

            return;
        }

        if (auto replyEvent = reqCtxPtr->MakeReplyEvent()) {
            Send(ReplyTo_, replyEvent.release());
        }

        if (0 == --RequestsCountdown_) {
            Send(ReplyTo_, reqCtxPtr->MakeCompletionEvent().release());

            PassAway();
        }
    }

    void OnPerformRequest(TEvPerformRequest::TPtr& ev) {
        PerformRequest(std::move(ev->Get()->ReqCtx));
    }

// Non-handlers
private:
    void PerformRequest(IRequestCtxPtr reqCtxPtr) {
        auto* actorSystem = TActorContext::ActorSystem();
        auto replyTo = SelfId();

        auto future = reqCtxPtr->PerformRequest();
        future.Subscribe([=, reqCtxPtr{std::move(reqCtxPtr)}](const NThreading::TFuture<void>&) mutable {
            // no need to check the future's state, 'cause reqCtx.PerformRequest() had already done that and has stored
            // a result
            actorSystem->Send(
                    replyTo,
                    new TEvResponse{std::move(reqCtxPtr)});
        });
    }

    void Start() {
        for (auto& reqCtxPtr: RequestContexts_) {
            if (Options_.ScatteringInterval) {
                TDuration delay = TDuration::MilliSeconds(RandomNumber(Options_.ScatteringInterval.MilliSeconds()));
                Schedule(delay, new TEvPerformRequest{reqCtxPtr});

                continue;
            }

            PerformRequest(reqCtxPtr);
        }
    }

private:
    TRequestsExecutorOptions Options_;
    // TODO(ivanzhukov): don't store the vector -> requests will be available through events
    TVector<IRequestCtxPtr> RequestContexts_;
    TActorId ReplyTo_;

    size_t RequestsCountdown_{0};
};

} // namespace

THolder<IActor> CreateRequestsExecutorActor(
        TRequestsExecutorOptions options,
        TVector<IRequestCtxPtr> reqCtxs,
        TActorId replyTo)
{
    return MakeHolder<TRequestsExecutorActor>(std::move(options), std::move(reqCtxs), replyTo);
}

} // namespace NSolomon
