#pragma once

#include <solomon/libs/cpp/actors/events/events.h>

#include <library/cpp/actors/core/actor_bootstrapped.h>
#include <library/cpp/actors/core/event_local.h>
#include <library/cpp/threading/future/future.h>

#include <memory>


namespace NSolomon {

/**
 * A single request object with info and actions
 */
class IRequestCtx: public std::enable_shared_from_this<IRequestCtx> {
public:
    virtual ~IRequestCtx() = default;

    /**
     * Sends a new request and returns a future that resolves whenever a response or an error is received
     */
    virtual NThreading::TFuture<void> PerformRequest() = 0;

    /**
     * Tells if a request can be and should be retried
     */
    virtual bool ShouldRetry() const = 0;

    /**
     * @returns Human-readable string representation of this request
     */
    virtual TString What() const = 0;

    /**
     * Tells how many requests have been sent from this context so far
     */
    virtual size_t TimesPerformed() const = 0;

    /**
     * Returns an event with a response after all retries.
     * Will be sent to the original recipient actor
     */
    virtual std::unique_ptr<NActors::IEventBase> MakeReplyEvent() = 0;

    // XXX(ivanzhukov): should it be inside _every_ IRequestCtx? Seems like it should be a separate entity
    /**
     * Returns an event that signals that all requests are completed.
     * This is the last event that will be sent to the original recipient actor
     */
    virtual std::unique_ptr<NActors::IEventBase> MakeCompletionEvent() = 0;
};

using IRequestCtxPtr = std::shared_ptr<IRequestCtx>;

struct TRequestsExecutorOptions {
    TRequestsExecutorOptions& SetMaxRetries(size_t maxRetries) {
        MaxTries = maxRetries + 1;
        return *this;
    }

    TRequestsExecutorOptions& SetTimeout(TDuration timeout) {
        Timeout = timeout;
        return *this;
    }

    TRequestsExecutorOptions& SetBackoff(TDuration backoff) {
        Backoff = backoff;
        return *this;
    }

    TRequestsExecutorOptions& SetScatteringInterval(TDuration interval) {
        ScatteringInterval = interval;
        return *this;
    }

    size_t MaxTries{1};
    TDuration Timeout;
    TDuration Backoff;
    TDuration ScatteringInterval;
};

THolder<NActors::IActor> CreateRequestsExecutorActor(
        TRequestsExecutorOptions options,
        TVector<IRequestCtxPtr> reqCtxs,
        NActors::TActorId replyTo);

} // namespace NSolomon
