#include "authentication_actor.h"
#include "authentication_cache.h"

#include <solomon/libs/cpp/logging/logging.h>

#include <library/cpp/actors/core/actor_bootstrapped.h>

using namespace NActors;

namespace NSolomon::NAuth {

namespace {

class THeadersAdapter: public NSolomon::IHeaders {
public:
    explicit THeadersAdapter(const NHttp::THeaders& headers)
        : Headers_(headers)
    {
    }

    TMaybe<TStringBuf> Find(TStringBuf key) const override {
        auto it = Headers_.Headers.find(key);
        if (it == Headers_.Headers.end()) {
            return {};
        }
        return it->second;
    }

    void Add(TStringBuf, TStringBuf) override {
        Y_FAIL("method not implemented");
    }

    void ForEach(std::function<void(TStringBuf, TStringBuf)> cb) const override {
        for (const auto&[key, value]: Headers_.Headers) {
            cb(key, value);
        }
    }

private:
    const NHttp::THeaders& Headers_;
};

class TAuthenticationActor: public TActorBootstrapped<TAuthenticationActor>, private TPrivateEvents {
private:
    enum {
        AuthenticationSuccess = SpaceBegin,
        AuthenticationFail
    };

    struct TAuthenticationSuccess : public NActors::TEventLocal<TAuthenticationSuccess, AuthenticationSuccess> {
        TActorId RequesterId;
        TAuthToken Token;
        TAuthSubject AuthSubject;

        TAuthenticationSuccess(TActorId requesterId, TAuthToken token, TAuthSubject authSubject)
            : RequesterId(requesterId)
            , Token(std::move(token))
            , AuthSubject(std::move(authSubject))
        {
        }
    };

    struct TAuthenticationFail : public NActors::TEventLocal<TAuthenticationFail, AuthenticationFail> {
        TActorId RequesterId;
        TAuthToken Token;
        TAuthError AuthError;

        TAuthenticationFail(TActorId requesterId, TAuthToken token, TAuthError authError)
            : RequesterId(requesterId)
            , Token(std::move(token))
            , AuthError(std::move(authError))
        {
        }
    };

public:
    explicit TAuthenticationActor(std::shared_ptr<IAuthenticator> authenticator)
        : Authenticator_(std::move(authenticator))
    {
    }

    void Bootstrap() {
        Become(&TAuthenticationActor::Main);
        Schedule(Cache_.GetMaxTTL(), new TEvents::TEvWakeup());
    }

    STATEFN(Main) {
        switch (ev->GetTypeRewrite()) {
            hFunc(TAuthEvents::TAuthenticationRequest, OnAuthenticationRequest)
            hFunc(TAuthenticationSuccess, OnAuthenticationSuccess)
            hFunc(TAuthenticationFail, OnAuthenticationFail)
            sFunc(TEvents::TEvWakeup, OnWakeUp)
            hFunc(TEvents::TEvPoison, OnPoison)
        }
    }

    void OnAuthenticationRequest(TAuthEvents::TAuthenticationRequest::TPtr& ev) {
        if (Poisoned_) {
            Send(ev->Sender, new TAuthEvents::TAuthenticationResponse{
                    TAuthError{
                            .Type = EAuthErrorType::NonRetriable,
                            .Message = "authentication actor is passed away"
                    }}, 0, ev->Cookie);
            return;
        }

        MON_DEBUG(Auth, "Start authentication request processing");
        THeadersAdapter headers(ev->Get()->Headers);
        auto token = Authenticator_->GetToken(&headers);
        if (token.Fail()) {
            MON_DEBUG(Auth, "Failed to get auth token. Error: " << token.Error().GetMessage());
            Send(ev->Sender, new TAuthEvents::TAuthenticationResponse(std::move(token.ExtractError())), 0, ev->Cookie);
            return;
        }

        MON_DEBUG(Auth, "Auth token: type: " << token.Value().Type);
        auto now = TActivationContext::Now();
        auto authResult = Cache_.Get(token.Value(), now);
        if (authResult) {
            if (authResult.value().Success()) {
                MON_DEBUG(Auth, "Successful authentication via cache. Type: " << token.Value().Type);
                Send(ev->Sender, new TAuthEvents::TAuthenticationResponse(authResult.value().Extract()), 0, ev->Cookie);
            } else {
                MON_DEBUG(Auth, "Failed authentication via cache. Type: " << token.Value().Type << "Error: " << authResult->Error().Message);
                Send(ev->Sender, new TAuthEvents::TAuthenticationResponse(authResult.value().ExtractError()), 0, ev->Cookie);
            }
            return;
        }

        Authenticate(token.Value(), ev->Sender, ev->Cookie);
    }

    void Authenticate(const TAuthToken& token, TActorId requesterId, ui64 cookie) {
        auto future = Authenticator_->Authenticate(token);
        ++Inflight_;
        auto system = NActors::TActorContext::ActorSystem();
        auto id = SelfId();
        future.Subscribe([system, id, cookie, requesterId, token](TAsyncAuthResult authResult) {
            TAuthResult res;
            try {
                res = authResult.ExtractValueSync();
            } catch (...) {
                TString errMessage = CurrentExceptionMessage();
                TAuthError error{.Type = EAuthErrorType::InternalError, .Message = std::move(errMessage)};
                auto authFail = std::make_unique<TAuthenticationFail>(requesterId, token, std::move(error));
                system->Send(new IEventHandle{id, id, authFail.release(), 0, cookie});
                return;
            }

            if (res.Success()) {
                auto authSuccess = std::make_unique<TAuthenticationSuccess>(requesterId, token, std::move(res.Extract()));
                system->Send(new IEventHandle{id, id, authSuccess.release(), 0, cookie});
            } else {
                auto authFail = std::make_unique<TAuthenticationFail>(requesterId, token, std::move(res.ExtractError()));
                system->Send(new IEventHandle{id, id, authFail.release(), 0, cookie});
            }
        });
    }

    void OnAuthenticationSuccess(TAuthenticationSuccess::TPtr& ev) {
        MON_DEBUG(Auth, "Authentication success. Type: " << ev->Get()->AuthSubject.GetAuthType());
        auto now = TActivationContext::Now();
        Cache_.Put(std::move(ev->Get()->Token), ev->Get()->AuthSubject, now);
        Send(ev->Get()->RequesterId, new TAuthEvents::TAuthenticationResponse(std::move(ev->Get()->AuthSubject)), 0, ev->Cookie);
        --Inflight_;
        MaybePassAway();
    }

    void OnAuthenticationFail(TAuthenticationFail::TPtr& ev) {
        if (ev->Get()->AuthError.Type == EAuthErrorType::InternalError) {
            MON_ERROR(Auth, "Authentication internal error: " << ev->Get()->AuthError.Message);
            TAuthError err{.Type = EAuthErrorType::NonRetriable, .Message = "internal error"};
            Send(ev->Get()->RequesterId, new TAuthEvents::TAuthenticationResponse(std::move(err)), 0, ev->Cookie);
        } else {
            MON_DEBUG(Auth, "Authentication failed. Error: " << ev->Get()->AuthError.Message);
            auto now = TActivationContext::Now();
            Cache_.Put(std::move(ev->Get()->Token), ev->Get()->AuthError, now);
            Send(ev->Get()->RequesterId, new TAuthEvents::TAuthenticationResponse(std::move(ev->Get()->AuthError)), 0, ev->Cookie);
        }
        --Inflight_;
        MaybePassAway();
    }

    void MaybePassAway() {
        if (Inflight_ == 0 && Poisoned_) {
            if (PoisonSender_) {
                Send(PoisonSender_, new TEvents::TEvPoisonTaken);
                PoisonSender_ = {};
            }
            PassAway();
        }
    }

    void OnWakeUp() {
        auto now = TActivationContext::Now();
        Cache_.EvictExpired(now);
        Schedule(Cache_.GetMaxTTL(), new TEvents::TEvWakeup());
    }

    void OnPoison(TEvents::TEvPoison::TPtr& ev) {
        if (Poisoned_) {
            return;
        }
        Poisoned_ = true;
        PoisonSender_ = ev->Sender;
        Send(ev->Sender, new TEvents::TEvPoisonTaken);
        MaybePassAway();
    }

private:
    std::shared_ptr<IAuthenticator> Authenticator_;
    TAuthenticationCache Cache_;
    size_t Inflight_{0};
    bool Poisoned_{false};
    TActorId PoisonSender_{};
};

} // namespace

std::unique_ptr<IActor> CreateAuthenticationActor(std::shared_ptr<IAuthenticator> authenticator) {
    return std::make_unique<TAuthenticationActor>(std::move(authenticator));
}

} // namespace NSolomon::NAuth
