#include "authentication_cache.h"

namespace NSolomon::NAuth {

const size_t POSITIVE_CACHE_MAX_SIZE = 6000;
const size_t NEGATIVE_CACHE_MAX_SIZE = 6000;
constexpr TDuration POSITIVE_CACHE_LIFE_TIME = TDuration::Minutes(5);
constexpr TDuration NEGATIVE_CACHE_LIFE_TIME = TDuration::Minutes(1);

TAuthenticationCacheParams TAuthenticationCacheParams::Default() {
    return TAuthenticationCacheParams {
            .PositiveCacheTTL = POSITIVE_CACHE_LIFE_TIME,
            .PositiveCacheSize = POSITIVE_CACHE_MAX_SIZE,
            .NegativeCacheTTL = NEGATIVE_CACHE_LIFE_TIME,
            .NegativeCacheSize = NEGATIVE_CACHE_MAX_SIZE
    };
}

TAuthenticationCache::TAuthenticationCache(TAuthenticationCacheParams params)
    : PositiveCache_(params.PositiveCacheTTL, params.PositiveCacheSize)
    , NegativeCache_(params.NegativeCacheTTL, params.NegativeCacheSize)
{
}

std::optional<TAuthResult> TAuthenticationCache::Get(const TAuthToken& token, TInstant now) {
    auto authSubject = PositiveCache_.Get(token, now);
    if (authSubject) {
        return TAuthResult::FromValue(authSubject.value());
    }

    auto authError = NegativeCache_.Get(token, now);
    if (authError) {
        return TAuthResult::FromError(authError.value());
    }

    return {};
}

void TAuthenticationCache::Put(TAuthToken token, TAuthResult authResult, TInstant now) {
    if (authResult.Success()) {
        PositiveCache_.Put(std::move(token), authResult.Extract(), std::move(now));
    } else {
        NegativeCache_.Put(std::move(token), authResult.ExtractError(), std::move(now));
    }
}

void TAuthenticationCache::EvictExpired(TInstant now) {
    PositiveCache_.EvictExpired(now);
    NegativeCache_.EvictExpired(now);
}

} // namespace NSolomon::NAuth
