#pragma once

#include <library/cpp/cache/cache.h>

#include <util/datetime/base.h>

#include <optional>

namespace NSolomon::NAuth {

template<typename TValue>
struct TExpireValue {
    TValue Value;
    TInstant ExpireAt;

    TExpireValue(TValue&& value, TInstant expireAt)
        : Value{value}
        , ExpireAt{expireAt}
    {
    }
};

template<typename TKey, typename TValue, typename TDeleter = TNoopDelete, class TSizeProvider = TUniformSizeProvider<TExpireValue<TValue>>, typename TAllocator = std::allocator<void>>
class TExpireLruCache: protected TCache<TKey, TExpireValue<TValue>, TLRUList<TKey, TExpireValue<TValue>, TSizeProvider>, TDeleter, TAllocator> {
protected:
    using TListType = TLRUList<TKey, TExpireValue<TValue>, TSizeProvider>;
    using TBase = TCache<TKey, TExpireValue<TValue>, TLRUList<TKey, TExpireValue<TValue>, TSizeProvider>, TDeleter, TAllocator>;

public:
    TExpireLruCache(TDuration lifeTime, size_t maxSize)
        : TBase(TListType(maxSize, TSizeProvider{}), /*multiValue = */false)
        , LifeTime_(lifeTime)
    {
    }

    std::optional<TValue> Get(const TKey& key, TInstant now) {
        auto it = TBase::FindWithoutPromote(key);
        if (it == TBase::End()) {
            return {};
        }

        if (it.Value().ExpireAt < now) {
            TBase::Erase(std::move(it));
            return {};
        }

        TValue value{it.Value().Value};
        Put(key, value, now);
        return value;
    }
    /**
     * return value for key if exists, makes no promotion
     */
    std::optional<TExpireValue<TValue>> Test(const TKey& key) const {
        auto it = TBase::FindWithoutPromote(key);
        if (it == TBase::End()) {
            return {};
        }

        return it.Value();
    }

    void Put(TKey key, TValue value, TInstant now) {
        TBase::Update(std::move(key), TExpireValue{std::move(value), now + LifeTime_});
    }

    void EvictExpired(TInstant now) {
        while (!TBase::Empty()) {
            const auto* item = TBase::List.GetOldest();
            if (item->Value.ExpireAt >= now) {
                break;
            }
            auto it = TBase::FindWithoutPromote(item->Key);
            TBase::Erase(std::move(it));
        }
    }

    /// Elements count
    size_t GetSize() const {
        return TBase::Size();
    }

    /// Sum of all elements sizes. Equals to GetSize() if TSizeProvider = TUniformSizeProvider<TValue>
    size_t GetTotalSize() const {
        return TBase::List.GetTotalSize();
    }

    /// Max possible value of GetTotalSize()
    size_t GetMaxSize() const {
        return TBase::List.GetMaxSize();
    }

    TDuration GetTTL() const {
        return LifeTime_;
    }

private:
    const TDuration LifeTime_;
};

} // namespace NSolomon::NAuth
