#include <solomon/libs/cpp/auth/actor/authentication_cache.h>
#include <solomon/libs/cpp/auth/core/authenticator.h>

#include <library/cpp/testing/gtest/gtest.h>

using namespace NSolomon::NAuth;

TEST(TAuthenticationCacheTest, ActualValidToken) {
    TAuthenticationCache cache;
    TAuthToken token(EAuthType::Iam, "abc-1234-xyz");
    TAuthSubject authSubject{.Subject = TFakeAuthSubject{.AuthType = token.Type}};
    TInstant now = TInstant::Now();

    cache.Put(token, std::move(authSubject), now);
    auto res = cache.Get(token, now + TDuration::Seconds(1));
    ASSERT_TRUE(res);
    ASSERT_TRUE(res.value().Success());
    ASSERT_EQ(token.Type, res.value().Value().GetAuthType());
}

TEST(TAuthenticationCacheTest, ActualFailedToken) {
    TAuthenticationCache cache;
    TAuthToken token(EAuthType::Iam, "abc-1234-xyz");
    TAuthError authError{.Type = EAuthErrorType::FailedAuth, .Message = "test error"};
    TInstant now = TInstant::Now();

    cache.Put(token, authError, now);
    auto res = cache.Get(token, now + TDuration::Seconds(1));
    ASSERT_TRUE(res);
    ASSERT_TRUE(res.value().Fail());
    ASSERT_EQ(authError.Type, res.value().Error().Type);
    ASSERT_EQ(authError.Message, res.value().Error().Message);
}

TEST(TAuthenticationCacheTest, EvictStaleValidToken) {
    TAuthenticationCache cache({
            .PositiveCacheTTL = TDuration::Minutes(1),
            .PositiveCacheSize = 10,
            .NegativeCacheTTL = TDuration::Minutes(1),
            .NegativeCacheSize = 10
    });
    TAuthToken token(EAuthType::Iam, "abc-1234-xyz");
    TAuthSubject authSubject{.Subject = TFakeAuthSubject{.AuthType = token.Type}};
    TInstant now = TInstant::Now();

    cache.Put(token, std::move(authSubject), now);
    now += TDuration::Minutes(2);
    cache.EvictExpired(now);
    auto res = cache.Get(token, now);
    ASSERT_FALSE(res);
}

TEST(TAuthenticationCacheTest, EvictStaleFailedToken) {
    TAuthenticationCache cache({
            .PositiveCacheTTL = TDuration::Minutes(1),
            .PositiveCacheSize = 10,
            .NegativeCacheTTL = TDuration::Minutes(1),
            .NegativeCacheSize = 10
    });
    TAuthToken token(EAuthType::Iam, "abc-1234-xyz");
    TAuthError authError{.Type = EAuthErrorType::FailedAuth, .Message = "test error"};
    TInstant now = TInstant::Now();

    cache.Put(token, std::move(authError), now);
    now += TDuration::Minutes(2);
    cache.EvictExpired(now);
    auto res = cache.Get(token, now);
    ASSERT_FALSE(res);
}

TEST(TAuthenticationCacheTest, EvictLRUValidToken) {
    const int cacheSize = 10;
    TAuthenticationCache cache({
            .PositiveCacheTTL = TDuration::Minutes(1),
            .PositiveCacheSize = cacheSize,
            .NegativeCacheTTL = TDuration::Minutes(1),
            .NegativeCacheSize = cacheSize
    });

    TInstant start = TInstant::Now();
    TAuthSubject authSubject{.Subject = TFakeAuthSubject{.AuthType = EAuthType::Iam}};
    for (int i = 0; i < cacheSize; i++) {
        TInstant tokenTime = start + TDuration::Seconds(i);
        TAuthToken token(EAuthType::Iam, "abc-1234-xyz" + ToString(i));
        cache.Put(std::move(token), authSubject, tokenTime);
    }

    TInstant now = start + TDuration::Seconds(cacheSize);
    for (int i = 0; i < cacheSize - 1; i++) {
        TAuthToken token(EAuthType::Iam, "abc-1234-xyz" + ToString(i));
        auto res = cache.Get(token, now);
        ASSERT_TRUE(res);
        ASSERT_TRUE(res.value().Success());
        ASSERT_EQ(token.Type, res.value().Value().GetAuthType());
    }

    TAuthToken evictingToken(EAuthType::Iam, "abc-1234-xyz" + ToString(cacheSize));
    cache.Put(evictingToken, authSubject, now);
    TAuthToken evictedToken(EAuthType::Iam, "abc-1234-xyz" + ToString(cacheSize - 1));
    ASSERT_FALSE(cache.Get(evictedToken, now));

    auto res = cache.Get(evictingToken, now);
    ASSERT_TRUE(res);
    ASSERT_TRUE(res.value().Success());
    ASSERT_EQ(evictingToken.Type, res.value().Value().GetAuthType());
}

TEST(TAuthenticationCacheTest, EvictLRUFailedToken) {
    const int cacheSize = 10;
    TAuthenticationCache cache({
            .PositiveCacheTTL = TDuration::Minutes(1),
            .PositiveCacheSize = cacheSize,
            .NegativeCacheTTL = TDuration::Minutes(1),
            .NegativeCacheSize = cacheSize
    });

    TInstant start = TInstant::Now();
    TAuthError authError{.Type = EAuthErrorType::FailedAuth, .Message = "test error"};
    for (int i = 0; i < cacheSize; i++) {
        TInstant tokenTime = start + TDuration::Seconds(i);
        TAuthToken token(EAuthType::Iam, "abc-1234-xyz" + ToString(i));
        cache.Put(std::move(token), authError, tokenTime);
    }

    TInstant now = start + TDuration::Seconds(cacheSize);
    for (int i = 0; i < cacheSize - 1; i++) {
        TAuthToken token(EAuthType::Iam, "abc-1234-xyz" + ToString(i));
        auto res = cache.Get(token, now);
        ASSERT_TRUE(res);
        ASSERT_TRUE(res.value().Fail());
        ASSERT_EQ(authError.Type, res.value().Error().Type);
        ASSERT_EQ(authError.Message, res.value().Error().Message);
    }

    TAuthToken evictingToken(EAuthType::Iam, "abc-1234-xyz" + ToString(cacheSize));
    cache.Put(evictingToken, authError, now);
    TAuthToken evictedToken(EAuthType::Iam, "abc-1234-xyz" + ToString(cacheSize - 1));
    ASSERT_FALSE(cache.Get(evictedToken, now));

    auto res = cache.Get(evictingToken, now);
    ASSERT_TRUE(res);
    ASSERT_TRUE(res.value().Fail());
    ASSERT_EQ(authError.Type, res.value().Error().Type);
    ASSERT_EQ(authError.Message, res.value().Error().Message);
}
