#include <solomon/libs/cpp/auth/actor/expire_lru_cache.h>

#include <library/cpp/testing/gtest/gtest.h>

#include <util/string/cast.h>

using namespace NSolomon::NAuth;

TEST(TExpireLruCacheTest, PutOne) {
    TExpireLruCache<int, TString> cache(TDuration::Minutes(1), 10);
    auto now = TInstant::Now();
    cache.Put(0, "0", now);
    auto res = cache.Get(0, now + TDuration::Seconds(1));
    ASSERT_TRUE(res);
    ASSERT_EQ("0", res.value());
}

TEST(TExpireLruCacheTest, GetFromEmpty) {
    TExpireLruCache<int, TString> cache(TDuration::Minutes(1), 10);
    auto now = TInstant::Now();
    auto res = cache.Get(0, now);
    ASSERT_FALSE(res);
}

TEST(TExpireLruCacheTest, EvictExpiredFromEmpty) {
    TExpireLruCache<int, TString> cache(TDuration::Minutes(1), 10);
    auto now = TInstant::Now();
    cache.EvictExpired(now);
    auto res = cache.Get(0, now);
    ASSERT_FALSE(res);
    ASSERT_EQ(0ul, cache.GetSize());
}

TEST(TExpireLruCacheTest, EvictExpired) {
    const int ttl = 3;
    const int evictCount = 4;
    const int count = 10;
    TExpireLruCache<int, TString> cache(TDuration::Minutes(ttl), 10);

    const auto start = TInstant::Now();
    for (int i = 0; i < count; i++) {
        cache.Put(i, ToString(i), start + TDuration::Minutes(i));
    }

    cache.EvictExpired(start + TDuration::Minutes(evictCount + ttl));
    ASSERT_EQ(static_cast<size_t>(count - evictCount), cache.GetSize());
    for (int i = 0; i < evictCount; i++) {
        auto res = cache.Test(i);
        ASSERT_FALSE(res);
    }

    for (int i = evictCount; i < count; i++) {
        auto res = cache.Test(i);
        ASSERT_TRUE(res);
        ASSERT_EQ(ToString(i), res.value().Value);
    }
}

TEST(TExpireLruCacheTest, EvictLRU) {
    const int ttl = 3;
    const int evictCount = 4;
    const int capacity = 10;
    const int count = capacity + evictCount;
    TExpireLruCache<int, TString> cache(TDuration::Minutes(ttl), capacity);

    const auto start = TInstant::Now();
    for (int i = 0; i < count; i++) {
        cache.Put(i, ToString(i), start + TDuration::Minutes(i));
    }

    ASSERT_EQ(static_cast<size_t>(capacity), cache.GetSize());
    for (int i = 0; i < evictCount; i++) {
        auto res = cache.Test(i);
        ASSERT_FALSE(res);
    }

    for (int i = evictCount; i < count; i++) {
        auto res = cache.Test(i);
        ASSERT_TRUE(res);
        ASSERT_EQ(ToString(i), res.value().Value);
    }
}

TEST(TExpireLruCacheTest, PromoteOnGet) {
    const int ttl = 100;
    const int capacity = 10;
    TExpireLruCache<int, TString> cache(TDuration::Minutes(ttl), capacity);

    const auto start = TInstant::Now();
    const auto end = start + TDuration::Minutes(capacity);
    for (int i = 0; i < capacity; i++) {
        cache.Put(i, ToString(i), start + TDuration::Minutes(i));
    }

    cache.Get(0, end);
    cache.Put(capacity, ToString(capacity), start + TDuration::Minutes(capacity));

    auto res0 = cache.Test(0);
    ASSERT_TRUE(res0);
    ASSERT_EQ(ToString(0), res0.value().Value);
    ASSERT_EQ(end + TDuration::Minutes(ttl), res0.value().ExpireAt);
    ASSERT_FALSE(cache.Test(1));

    for (int i = 2; i <= capacity; i++) {
        auto res = cache.Test(i);
        ASSERT_TRUE(res);
        ASSERT_EQ(ToString(i), res.value().Value);
    }
}

TEST(TExpireLruCacheTest, PutUpdateValueAndExpire) {
    const int ttl = 100;
    const int capacity = 10;
    TExpireLruCache<int, TString> cache(TDuration::Minutes(ttl), capacity);

    const auto start = TInstant::Now();
    const auto end = start + TDuration::Minutes(capacity);
    for (int i = 0; i < capacity; i++) {
        cache.Put(i, ToString(i), start + TDuration::Minutes(i));
    }
    cache.Put(0, ToString("zero"), end);

    auto res0 = cache.Test(0);
    ASSERT_TRUE(res0);
    ASSERT_EQ("zero", res0.value().Value);
    ASSERT_EQ(end + TDuration::Minutes(ttl), res0.value().ExpireAt);
}
