#pragma once

#include "auth_types.h"

#include <solomon/libs/cpp/cloud/access_service/access_service.h>

#include <library/cpp/tvmauth/checked_user_ticket.h>
#include <library/cpp/tvmauth/checked_service_ticket.h>
#include <library/cpp/tvmauth/type.h>

namespace NSolomon::NAuth {

class TIamSubject {
public:
    explicit TIamSubject(TIamAccount account)
        : Account_(std::move(account))
    {
    }

    bool IsUserAccount() const {
        return Account_.Type == EIamAccountType::User;
    }

    bool IsServiceAccount() const {
        return Account_.Type == EIamAccountType::Service;
    }

    const TString& GetId() const {
        return Account_.Id;
    }

    const TString& GetFolderId() const {
        if (IsServiceAccount()) {
            return Account_.ScopeId;
        }
        return GetEmptyId();
    }

    const TString& GetFederationId() const {
        if (IsUserAccount()) {
            return Account_.ScopeId;
        }
        return GetEmptyId();
    };

private:
    static const TString& GetEmptyId();

private:
    TIamAccount Account_;
};

template<typename T>
class TTvmTicket {
public:

    TTvmTicket(T&& ticket)
        : Impl_(new TTicketHolder(std::move(ticket)))
    {
    }

    TTvmTicket() = default;

    const T& Ticket() const {
        return Impl_->Ticket;
    }

private:
    struct TTicketHolder {
        T Ticket;

        TTicketHolder(T&& ticket)
            : Ticket(std::move(ticket))
        {
        }
    };

private:
    std::shared_ptr<TTicketHolder> Impl_;
};

using TTvmUserTicket = TTvmTicket<NTvmAuth::TCheckedUserTicket>;
using TTvmServiceTicket = TTvmTicket<NTvmAuth::TCheckedServiceTicket>;

class TTvmSubject {
public:
    explicit TTvmSubject(NTvmAuth::TCheckedUserTicket&& checkedUserTicket)
        : CheckedTicket_(TTvmUserTicket(std::move(checkedUserTicket)))
    {
    }

    explicit TTvmSubject(NTvmAuth::TCheckedServiceTicket&& checkedServiceTicket)
        : CheckedTicket_(TTvmServiceTicket(std::move(checkedServiceTicket)))
    {
    }

    bool IsUserAccount() const {
        return std::holds_alternative<TTvmUserTicket>(CheckedTicket_);
    }

    bool IsServiceAccount() const {
        return std::holds_alternative<TTvmServiceTicket>(CheckedTicket_);
    }

    const NTvmAuth::TCheckedUserTicket* GetUserTicket() const {
        return &std::get_if<TTvmUserTicket>(&CheckedTicket_)->Ticket();
    }

    const NTvmAuth::TCheckedServiceTicket* GetServiceTicket() const {
        return &std::get_if<TTvmServiceTicket>(&CheckedTicket_)->Ticket();
    }

private:
    std::variant<TTvmUserTicket, TTvmServiceTicket> CheckedTicket_;
};

/// for tests
struct TFakeAuthSubject {
    EAuthType AuthType;
};

struct TAuthSubject {
    std::variant<TIamSubject, TTvmSubject, TFakeAuthSubject> Subject;

    EAuthType GetAuthType() const {
        if (std::holds_alternative<TIamSubject>(Subject)) {
            return EAuthType::Iam;
        }
        if (std::holds_alternative<TTvmSubject>(Subject)) {
            return std::get<TTvmSubject>(Subject).IsServiceAccount() ? EAuthType::TvmService : EAuthType::TvmUser;
        }
        if (Y_UNLIKELY(std::holds_alternative<TFakeAuthSubject>(Subject))) {
            return std::get<TFakeAuthSubject>(Subject).AuthType;
        }
        Y_FAIL("Unknown auth subject type");
    }

    const TIamSubject& AsIam() const {
        return std::get<TIamSubject>(Subject);
    }

    const TTvmSubject& AsTvm() const {
        return std::get<TTvmSubject>(Subject);
    }
};

}
