#pragma once

#include "auth_types.h"
#include "auth_subjects.h"

#include <solomon/libs/cpp/auth/core/auth_types.h_serialized.h>
#include <solomon/libs/cpp/auth/tvm/tvm.h>
#include <solomon/libs/cpp/cloud/access_service/access_service.h>
#include <solomon/libs/cpp/error_or/error_or.h>
#include <solomon/libs/cpp/http/client/http.h>
#include <solomon/libs/cpp/actors/events/events.h>
#include <solomon/libs/cpp/actors/events/slots.h>

#include <library/cpp/actors/core/event_local.h>
#include <library/cpp/actors/http/http.h>
#include <library/cpp/tvmauth/checked_user_ticket.h>

namespace NSolomon::NAuth {

enum class ETokenParseError {
    NoAuthHeader,
    UnknownAuthType,
    InvalidFormat
};

struct TTokenParseError {
    ETokenParseError Type;

    TString GetMessage() const;

    explicit TTokenParseError(ETokenParseError type)
        : Type(type)
    {
    }
};

struct TAuthToken {
    EAuthType Type;
    TString Value;

    TAuthToken(EAuthType type, TString value)
        : Type(type)
        , Value(std::move(value))
    {
    }

    bool operator==(const TAuthToken& other) const {
        return Type == other.Type && Value == other.Value;
    }
    bool operator!=(const TAuthToken& other) const {
        return !(*this == other);
    }
};

using TTokenParseResult = TErrorOr<TAuthToken, TTokenParseError>;

using TAuthTypeSet = std::unordered_set<EAuthType>;

enum class EAuthErrorType {
    Retriable,
    NonRetriable,
    FailedAuth,
    InternalError
};

struct TAuthError {
    EAuthErrorType Type;
    TString Message;
};

using TAuthResult = TErrorOr<TAuthSubject, TAuthError>;
using TAsyncAuthResult = NThreading::TFuture<TAuthResult>;

class IAuthenticator {
public:
    virtual ~IAuthenticator() = default;
    virtual TTokenParseResult GetToken(const IHeaders* httpHeaders) = 0;
    virtual TAsyncAuthResult Authenticate(const TAuthToken& token) = 0;
    virtual const TAuthTypeSet& GetTypes() const = 0;
};

using IAuthenticatorPtr = std::shared_ptr<IAuthenticator>;

IAuthenticatorPtr CreateIamAuthenticator(IAccessServiceClientPtr iamServiceClient);

IAuthenticatorPtr CreateServiceTvmAuthenticator(NTvm::ITvmClientPtr tvmClient);

IAuthenticatorPtr CreateUserTvmAuthenticator(NTvm::ITvmClientPtr tvmClient, ::NTvmAuth::EBlackboxEnv blackboxEnv);

/// for tests
IAuthenticatorPtr CreateFakeAuthenticator(EAuthType authType);

IAuthenticatorPtr CreateAuthenticatorMultiplexer(const TVector<IAuthenticatorPtr>& authenticators);

} // namespace NSolomon::NAuth
