#include "internal_authorizer.h"

#include <library/cpp/containers/absl_flat_hash/flat_hash_set.h>
#include <library/cpp/tvmauth/ticket_status.h>

#include <util/string/cast.h>

namespace NSolomon::NAuth {

namespace {

template<typename T>
struct TAccessSet {
    absl::flat_hash_set<T> Allowed;
    absl::flat_hash_set<T> Prohibited;

    bool IsAllowed(const T& id) const {
        return Allowed.contains(id) && !Prohibited.contains(id);
    }
};

NTvmAuth::TTvmId GetTvmId(const TString& id) {
    NTvmAuth::TTvmId tvmId{};
    Y_ENSURE(TryFromString(id, tvmId), "invalid TVM ID: " << id);
    return tvmId;
}

class TInternalAuthorizer: public IInternalAuthorizer {
public:
    explicit TInternalAuthorizer(const TInternalAccess& accessList) {
        for (int i = 0; i < accessList.allowed_size(); i++){
            const auto& entry = accessList.allowed(i);
            switch (entry.authtype()) {
                case TAccessListEntry_EInternalAuthType_TVM:
                    TvmAccessSet_.Allowed.insert(GetTvmId(entry.id()));
                    break;
                case TAccessListEntry_EInternalAuthType_IAM:
                    IamAccessSet_.Allowed.insert(entry.id());
                    break;
                default:
                    break;
            }
        }
        for (int i = 0; i < accessList.prohibited_size(); i++) {
            const auto& entry = accessList.prohibited(i);
            switch (entry.authtype()) {
                case TAccessListEntry_EInternalAuthType_TVM:
                    TvmAccessSet_.Prohibited.insert(GetTvmId(entry.id()));
                    break;
                case TAccessListEntry_EInternalAuthType_IAM:
                    IamAccessSet_.Prohibited.insert(entry.id());
                    break;
                default:
                    break;
            }
        }
    }

    bool IsAllowed(const TAuthSubject& authSubject) const override {
        const auto authType = authSubject.GetAuthType();
        switch (authType) {
            case EAuthType::Unknown:
            case EAuthType::OAuth:
                return false;
            case EAuthType::Iam:
                return IamAccessSet_.IsAllowed(authSubject.AsIam().GetId());
            case EAuthType::TvmService: {
                const auto* ticket = authSubject.AsTvm().GetServiceTicket();
                if (ticket->GetStatus() != NTvmAuth::ETicketStatus::Ok) {
                    return false;
                }
                return TvmAccessSet_.IsAllowed(ticket->GetSrc());
            }
            case EAuthType::TvmUser:
                /// only TVM services supported by internal authorizer
                return false;
        }
        return false;
    }

private:
    TAccessSet<TString> IamAccessSet_;
    TAccessSet<NTvmAuth::TTvmId> TvmAccessSet_;
};

} // namespace

IInternalAuthorizerPtr CreateInternalAuthorizer(const TInternalAccess& accessList) {
    return std::make_shared<TInternalAuthorizer>(accessList);
}

} // namespace NSolomon::NAuth
