#include <solomon/libs/cpp/auth/core/authenticator.h>
#include <solomon/libs/cpp/http/client/http.h>

#include <library/cpp/testing/gtest/gtest.h>

using namespace NSolomon::NAuth;

class TAuthenticatorTest: public ::testing::Test {
public:
    void SetUp() override {
        const TVector<EAuthType> authTypes{
            EAuthType::OAuth,
            EAuthType::Iam,
            EAuthType::TvmService,
            EAuthType::TvmUser};

        TVector<IAuthenticatorPtr> authenticators(::Reserve(authTypes.size()));
        for (auto type: authTypes) {
            authenticators.emplace_back(CreateFakeAuthenticator(type));
        }
        Authenticator_ = CreateAuthenticatorMultiplexer(authenticators);
    }

protected:
    void AuthenticateFromHeaders(const NSolomon::IHeaders* headers, EAuthType authType) {
        auto token = Authenticator_->GetToken(headers);
        ASSERT_TRUE(token.Success()) << token.Error().GetMessage();
        ASSERT_EQ(authType, token.Value().Type);

        auto authResult = Authenticator_->Authenticate(token.Value()).ExtractValueSync();
        ASSERT_TRUE(authResult.Success()) << authResult.Error().Message;
    }

    void ExpectTokenParseError(const NSolomon::IHeaders* headers, ETokenParseError expectedError) {
        auto token = Authenticator_->GetToken(headers);
        ASSERT_TRUE(token.Fail());
        ASSERT_EQ(expectedError, token.Error().Type);
    }

protected:
    IAuthenticatorPtr Authenticator_;
};

class TMockHeaders: public NSolomon::IHeaders {
public:
    TMockHeaders(const TVector<std::pair<TStringBuf, TStringBuf>>& headers) {
        for (const auto& [key, value]: headers) {
            Add(key, value);
        }
    }

    TMaybe<TStringBuf> Find(TStringBuf key) const override {
        auto it = Headers_.find(key);
        if (it == Headers_.end()) {
            return {};
        }
        return it->second;
    }

    void Add(TStringBuf key, TStringBuf value) override {
        Headers_[key] = value;
    }

    void ForEach(std::function<void(TStringBuf, TStringBuf)> f) const override {
        for (auto [key, value]: Headers_) {
            f(key, value);
        }
    }

private:
    TMap<TStringBuf, TStringBuf> Headers_;
};

TEST_F(TAuthenticatorTest, TvmServiceTicket) {
    const TMockHeaders headers{{
        {"X-Ya-Service-Ticket", "abc-1234-xyz"}
    }};
    AuthenticateFromHeaders(&headers, EAuthType::TvmService);
}

TEST_F(TAuthenticatorTest, TvmServiceAuthorizationTicket) {
    const TMockHeaders headers{{
        {"Authorization", "X-Ya-Service-Ticket abc-1234-xyz"}
    }};
    ExpectTokenParseError(&headers, ETokenParseError::NoAuthHeader);
}

TEST_F(TAuthenticatorTest, TvmServiceDuplicatedTicket) {
    const TMockHeaders headers{{
            {"X-Ya-Service-Ticket", "abc-1234-xyz"},
            {"Authorization", "X-Ya-Service-Ticket abc-1234-xyz"}
    }};
    AuthenticateFromHeaders(&headers, EAuthType::TvmService);
}

TEST_F(TAuthenticatorTest, TvmServiceAmbiguousTicket) {
    const TMockHeaders headers{{
            {"X-Ya-Service-Ticket", "abc-1234-xyz"},
            {"Authorization", "X-Ya-Service-Ticket xxx-1234-xyz"}
    }};
    auto token = Authenticator_->GetToken(&headers);
    ASSERT_TRUE(token.Success());
    ASSERT_EQ(EAuthType::TvmService, token.Value().Type);
    ASSERT_EQ("abc-1234-xyz", token.Value().Value);
}

TEST_F(TAuthenticatorTest, AmbiguousTicket1) {
    const TMockHeaders headers{{
            {"X-Ya-Service-Ticket", "abc-1234-xyz"},
            {"Authorization", "X-Ya-User-Ticket xxx-1234-xyz"}
    }};
    AuthenticateFromHeaders(&headers, EAuthType::TvmService);
}

TEST_F(TAuthenticatorTest, AmbiguousTicket2) {
    /**
     * Priority of tokens according the order in enum EAuthType
     */
    const TMockHeaders headers{{
            {"X-Ya-User-Ticket", "abc-1234-xyz"},
            {"X-Ya-Service-Ticket", "abc-1234-xyz"},
    }};
    AuthenticateFromHeaders(&headers, EAuthType::TvmService);
}

TEST_F(TAuthenticatorTest, TvmUserTicket) {
    const TMockHeaders headers{{
        {"X-Ya-User-Ticket", "abc-1234-xyz"}
    }};
    AuthenticateFromHeaders(&headers, EAuthType::TvmUser);
}

TEST_F(TAuthenticatorTest, TvmUserAuthorizationTicket) {
    const TMockHeaders headers{{
        {"Authorization", "X-Ya-User-Ticket abc-1234-xyz"}
    }};
    ExpectTokenParseError(&headers, ETokenParseError::NoAuthHeader);
}

TEST_F(TAuthenticatorTest, IamTicket) {
    const TMockHeaders headers{{
            {"Bearer", "abc-1234-xyz"}
    }};
    ExpectTokenParseError(&headers, ETokenParseError::NoAuthHeader);
}

TEST_F(TAuthenticatorTest, IamAuthorizationTicket) {
    const TMockHeaders headers{{
            {"Authorization", "Bearer abc-1234-xyz"}
    }};
    AuthenticateFromHeaders(&headers, EAuthType::Iam);
}

TEST_F(TAuthenticatorTest, OneWordAuthorizationTicket) {
    const TMockHeaders headers{{
            {"Authorization", "abc-1234-xyz"}
    }};
    ExpectTokenParseError(&headers, ETokenParseError::InvalidFormat);
}

TEST_F(TAuthenticatorTest, ThreeWordAuthorizationTicket) {
    const TMockHeaders headers{{
            {"Authorization", "Bearer abc-1234-xyz third_word"}
    }};
    ExpectTokenParseError(&headers, ETokenParseError::InvalidFormat);
}

TEST_F(TAuthenticatorTest, UnknownAuthorizationTicket) {
    const TMockHeaders headers{{
            {"Authorization", "xyz abc-1234-xyz"}
    }};
    ExpectTokenParseError(&headers, ETokenParseError::NoAuthHeader);

    auto iamAuthenticator = CreateFakeAuthenticator(EAuthType::Iam);
    auto token = iamAuthenticator->GetToken(&headers);
    ASSERT_TRUE(token.Fail());
    ASSERT_EQ(ETokenParseError::UnknownAuthType, token.Error().Type);
}
