#include <solomon/libs/cpp/auth/core/internal_authorizer.h>

#include <library/cpp/testing/gtest/gtest.h>
#include <library/cpp/protobuf/util/pb_io.h>
#include <library/cpp/tvmauth/unittest.h>

#include <util/stream/mem.h>

using namespace NSolomon;
using namespace NSolomon::NAuth;

class TInternalAuthorizerTest: public ::testing::Test {
public:
    void SetUp() override {
        InternalAccess_ = LoadAccessConfig(WhiteAndBlackConfig_);
        InternalAuthorizer_ = CreateInternalAuthorizer(InternalAccess_);
    }

protected:
    static TInternalAccess LoadAccessConfig(TStringBuf config) {
        TMemoryInput input(config);
        return ParseFromTextFormat<TInternalAccess>(input);
    }

    static TAuthSubject CreateIamServiceSubject(TString id) {
        auto iamSubject = TAuthSubject{
                .Subject = TIamSubject(
                        TIamAccount{
                            .Type = EIamAccountType::Service,
                            .Id = std::move(id)})};

        return iamSubject;
    }

    static TAuthSubject CreateTvmServiceSubject(NTvmAuth::TTvmId id) {
        auto ticket = NTvmAuth::NUnittest::CreateServiceTicket(NTvmAuth::ETicketStatus::Ok, id);
        auto tvmSubject = TAuthSubject{.Subject = TTvmSubject(std::move(ticket))};
        return tvmSubject;
    }

protected:
    static const TString WhiteAndBlackConfig_;
    TInternalAccess InternalAccess_;
    IInternalAuthorizerPtr InternalAuthorizer_;
};

const TString TInternalAuthorizerTest::WhiteAndBlackConfig_(
        R"(
            Allowed { AuthType: TVM, Id: "123456" }
            Allowed { AuthType: IAM, Id: "abc-1234-xyz" }
            Prohibited: [
                { AuthType: IAM, Id: "abc-1234-xyz-black" },
                { AuthType: TVM, Id: "999999" }
            ]
        )"
);

const TString WhiteIamId("abc-1234-xyz");
const TString BlackIamId("abc-1234-xyz-black");
const TString UnlistedIamId("abc-1234-xyz-unlisted");
const NTvmAuth::TTvmId WhiteTvmId = 123456u;
const NTvmAuth::TTvmId BlackTvmId = 999999u;
const NTvmAuth::TTvmId UnlistedTvmId = 777888u;

TEST_F(TInternalAuthorizerTest, EmptyAccessListsTest) {
    auto config = LoadAccessConfig("");
    auto authorizer = CreateInternalAuthorizer(config);
    auto iamSubject = CreateIamServiceSubject(WhiteIamId);
    ASSERT_FALSE(authorizer->IsAllowed(std::move(iamSubject))) << "expect authorization is not allowed with empty white list";

    auto tvmSubject = CreateTvmServiceSubject(WhiteTvmId);
    ASSERT_FALSE(authorizer->IsAllowed(std::move(tvmSubject))) << "expect authorization is not allowed with empty white list";
}

TEST_F(TInternalAuthorizerTest, LoadAccessListsTest) {
    ASSERT_EQ(2, InternalAccess_.allowed_size());
    ASSERT_EQ(TAccessListEntry_EInternalAuthType_TVM, InternalAccess_.allowed(0).authtype());
    ASSERT_EQ("123456", InternalAccess_.allowed(0).id());
    ASSERT_EQ(TAccessListEntry_EInternalAuthType_IAM, InternalAccess_.allowed(1).authtype());
    ASSERT_EQ("abc-1234-xyz", InternalAccess_.allowed(1).id());

    ASSERT_EQ(2, InternalAccess_.prohibited_size());
    ASSERT_EQ(TAccessListEntry_EInternalAuthType_IAM, InternalAccess_.prohibited(0).authtype());
    ASSERT_EQ("abc-1234-xyz-black", InternalAccess_.prohibited(0).id());
    ASSERT_EQ(TAccessListEntry_EInternalAuthType_TVM, InternalAccess_.prohibited(1).authtype());
    ASSERT_EQ("999999", InternalAccess_.prohibited(1).id());
}

TEST_F(TInternalAuthorizerTest, IamWhiteListedServiceTest) {
    auto iamSubject = CreateIamServiceSubject(WhiteIamId);
    ASSERT_TRUE(InternalAuthorizer_->IsAllowed(std::move(iamSubject))) << "white listed IAM Service Id is not allowed";
}

TEST_F(TInternalAuthorizerTest, IamBlackListedServiceTest) {
    auto iamSubject = CreateIamServiceSubject(BlackIamId);
    ASSERT_FALSE(InternalAuthorizer_->IsAllowed(std::move(iamSubject))) << "black listed IAM Service Id is allowed";
}

TEST_F(TInternalAuthorizerTest, IamUnlistedServiceTest) {
    auto iamSubject = CreateIamServiceSubject(UnlistedIamId);
    ASSERT_FALSE(InternalAuthorizer_->IsAllowed(std::move(iamSubject))) << "unlisted IAM Service Id is allowed";
}

TEST_F(TInternalAuthorizerTest, TvmWhiteListedServiceTest) {
    auto tvmSubject = CreateTvmServiceSubject(WhiteTvmId);
    ASSERT_TRUE(InternalAuthorizer_->IsAllowed(std::move(tvmSubject))) << "white listed TVM Service Id is not allowed";
}

TEST_F(TInternalAuthorizerTest, TvmBlackListedServiceTest) {
    auto tvmSubject = CreateTvmServiceSubject(BlackTvmId);
    ASSERT_FALSE(InternalAuthorizer_->IsAllowed(std::move(tvmSubject))) << "black listed TVM Service Id is allowed";
}

TEST_F(TInternalAuthorizerTest, TvmUnlistedServiceTest) {
    auto tvmSubject = CreateTvmServiceSubject(UnlistedTvmId);
    ASSERT_FALSE(InternalAuthorizer_->IsAllowed(std::move(tvmSubject))) << "unlisted TVM Service Id is allowed";
}
