#include "tvm.h"

#include <solomon/libs/cpp/logging/logging.h>
#include <solomon/libs/cpp/sync/rw_lock.h>

#include <library/cpp/tvmauth/client/facade.h>
#include <library/cpp/tvmauth/client/misc/api/dynamic_dst/tvm_client.h>

#include <library/cpp/actors/core/actorsystem.h>
#include <library/cpp/actors/core/log.h>

#include <library/cpp/monlib/metrics/metric_registry.h>
#include <library/cpp/monlib/metrics/timer.h>

#include <library/cpp/threading/future/future.h>
#include <library/cpp/threading/hot_swap/hot_swap.h>

#include <util/system/hp_timer.h>
#include <util/generic/algorithm.h>

using namespace NActors;
using namespace NTvmAuth;
using namespace NThreading;
using namespace NMonitoring;

namespace NSolomon::NAuth::NTvm {
namespace {
    struct TActorLogger final: public NTvmAuth::ILogger {
        TActorLogger(TActorSystem& actorSystem)
            : ActorSystem_{actorSystem}
        {
        }

        void Log(int severity, const TString& msg) override {
            MON_LOG_C(ActorSystem_, TvmClient, NLog::EPriority(severity), msg);
        }

    private:
        TActorSystem& ActorSystem_;
    };

    class TDynamicTvmClient: public NDynamicClient::TTvmClient {
    public:
        static TIntrusivePtr<TDynamicTvmClient> Create(const NTvmApi::TClientSettings& settings, TLoggerPtr logger) {
            TIntrusivePtr<TDynamicTvmClient> p(new TDynamicTvmClient(settings, std::move(logger)));
            p->Init();
            p->StartWorker();
            return p;
        }

    protected:
        TDynamicTvmClient(const NTvmApi::TClientSettings& settings, TLoggerPtr logger)
            : TTvmClient(settings, std::move(logger))
        {
        }
    };

    class TUpdatableTvmClient final: public ITvmClient {
    public:
        TUpdatableTvmClient(const NTvmApi::TClientSettings& settings, TLoggerPtr logger)
            : DynamicClient_(TDynamicTvmClient::Create(std::move(settings), std::move(logger)))
            , TvmClient_(new NTvmAuth::TTvmClient{TAsyncUpdaterPtr(DynamicClient_)})
        {
        }

        NThreading::TFuture<NTvmAuth::NDynamicClient::TAddResponse> Add(NTvmAuth::NDynamicClient::TDsts&& dsts) {
            return DynamicClient_->Add(std::move(dsts));
        }

        TString GetServiceTicketFor(const TTvmId dst) const override {
            return TvmClient_->GetServiceTicketFor(dst);
        }

        TCheckedServiceTicket CheckServiceTicket(TStringBuf ticket) const override {
            return TvmClient_->CheckServiceTicket(ticket);
        }

        TCheckedUserTicket CheckUserTicket(TStringBuf ticket, TMaybe<EBlackboxEnv> overrideEnv) const override {
            return TvmClient_->CheckUserTicket(ticket, overrideEnv);
        }

    private:
        TIntrusivePtr<TDynamicTvmClient> DynamicClient_;
        THolder<NTvmAuth::TTvmClient> TvmClient_;
    };

    class TUpdatableTicketProvider final: public NAuth::NTvm::ITicketProvider {
        struct TCounters {
            TCounters(TMetricRegistry& registry)
                : Inflight{registry.IntGauge({{"sensor", "tvm.addInFlight"}})}
                , AddFail{registry.Rate({{"sensor", "tvm.addFail"}})}
            {
            }

            TIntGauge* Inflight{nullptr};
            TRate* AddFail{nullptr};
        };

    public:
        TUpdatableTicketProvider(const TTicketProviderConfig& config, TLoggerPtr logger, TIntrusivePtr<TUpdatableTvmClient> tvmClient)
            : Logger_{std::move(logger)}
            , Client_{std::move(tvmClient)}
            , Counters_{config.Registry}
        {
            auto tvmConfig = config.Config;

            auto& cfgDest = tvmConfig.GetKnownTvmDestinations();
            TVector<NTvmAuth::NTvmApi::TClientSettings::TDst> clients(std::begin(cfgDest), std::end(cfgDest));
            {
                auto knownDest = KnownDestinations_.Write();
                knownDest->insert(std::begin(cfgDest), std::end(cfgDest));
            }
        }

        TTicketOrError GetTicket(NTvmAuth::TTvmId dst) const noexcept override {
            try {
                return Client_->GetServiceTicketFor(dst);
            } catch (...) {
                return TTicketOrError::FromError(CurrentExceptionMessage());
            }
        }

        void AddDestinationIds(TVector<ui32> ids) noexcept override {
            AddDestinationIdsImpl(ids.begin(), ids.end());
        }

        void AddDestinationId(ui32 id) noexcept override {
            ui32* it = &id;
            AddDestinationIdsImpl(it, it + 1);
        }

    private:
        template <typename TIterator>
        void AddDestinationIdsImpl(TIterator begin, TIterator end) {
            TIterator newEnd;
            {
                auto knownDestinations = KnownDestinations_.Read();
                newEnd = std::remove_if(begin, end, [&] (auto&& id) {
                    return knownDestinations->contains(id);
                });
            }

            if (std::distance(begin, newEnd) == 0) {
                return;
            }

            NDynamicClient::TDsts newIds;
            {
                auto knownDestinations = KnownDestinations_.Write();

                bool hasNew{false};
                for (auto it = begin; it != newEnd; ++it) {
                    auto [_, inserted] = knownDestinations->emplace(*it);
                    hasNew |= inserted;
                    newIds.emplace(*it);
                }

                if (!hasNew) {
                    return;
                }
            }

            Counters_.Inflight->Inc();
            Client_->Add(std::move(newIds)).Subscribe([this] (auto f) {
                Counters_.Inflight->Dec();
                HandleAdd(std::move(f));
            });
        }

        void HandleAdd(NThreading::TFuture<NDynamicClient::TAddResponse>&& f) try {
            auto val = f.ExtractValue();

            for (auto&& [dst, response]: val) {
                if (response.Status == NDynamicClient::EDstStatus::Fail) {
                    Logger_->Error(TStringBuilder() << "Cannot get ticket for " << dst.Id << ": " << response.Error);
                    Counters_.AddFail->Inc();
                    continue;
                }
            }
        } catch (...) {
            Logger_->Error(TStringBuilder() << "Ticket updater returned an error: " << CurrentExceptionMessage());
        }

    private:
        TLoggerPtr Logger_;
        TIntrusivePtr<TUpdatableTvmClient> Client_;
        NSync::TLightRwLock<THashSet<ui32>> KnownDestinations_;
        TCounters Counters_;
    };

    class TTvmServicesFactory: public ITvmServicesFactory {
    public:
        ITvmClientPtr CreateTvmClient() override {
            if (!TvmClient_) {
                TvmClient_ = new TUpdatableTvmClient(Settings_, Logger_);
            }
            return TvmClient_;
        }

        ITicketProviderPtr CreateTicketProvider() override {
            if (!TicketProvider_) {
                CreateTvmClient();
                TicketProvider_ = new TUpdatableTicketProvider(Config_, Logger_, TvmClient_);
            }
            return TicketProvider_;
        }

        explicit TTvmServicesFactory(TTicketProviderConfig config)
            : Config_(std::move(config))
            , Settings_(CreateClientSettings(Config_))
            , Logger_(::MakeIntrusive<TActorLogger>(Config_.ActorSystem))
        {
        }

    private:
        static NTvmApi::TClientSettings CreateClientSettings(const TTicketProviderConfig& config) {
            auto tvmConfig = config.Config;

            auto& cfgDest = tvmConfig.GetKnownTvmDestinations();
            TVector<NTvmAuth::NTvmApi::TClientSettings::TDst> clients(std::begin(cfgDest), std::end(cfgDest));

            TStringBuf secret(tvmConfig.GetSecret());

            NTvmApi::TClientSettings settings;
            if (auto&& cacheDir = tvmConfig.GetCacheDir(); !cacheDir.empty()) {
                settings.SetDiskCacheDir(tvmConfig.GetCacheDir());
            }

            settings.SetSelfTvmId(tvmConfig.GetSelfId());
            settings.EnableServiceTicketsFetchOptions(secret, std::move(clients));
            settings.EnableServiceTicketChecking();
            return settings;
        }

    private:
        const TTicketProviderConfig Config_;
        const NTvmApi::TClientSettings Settings_;
        TLoggerPtr Logger_;
        TIntrusivePtr<TUpdatableTvmClient> TvmClient_;
        TIntrusivePtr<TUpdatableTicketProvider> TicketProvider_;
    };

} // namespace

TIntrusivePtr<ITvmServicesFactory> ITvmServicesFactory::Create(TTicketProviderConfig config) {
    return MakeIntrusive<TTvmServicesFactory>(std::move(config));
}

} // namespace NSolomon::NAuth::NTvm
