#pragma once

#include <util/datetime/base.h>

namespace NSolomon {
namespace NPrivate {

/**
 * Base class for all implementations. Do not use it directly.
 */
class TBackoffBase {
protected:
    constexpr TBackoffBase(TDuration min, TDuration max) noexcept
        : Min_{min}
        , Max_{max}
    {
        Y_ASSERT(min > TDuration::Zero());
        Y_ASSERT(max > min);
    }

public:
    constexpr TDuration Min() const noexcept {
        return Min_;
    }

    constexpr TDuration Max() const noexcept {
        return Max_;
    }

protected:
    const TDuration Min_;
    const TDuration Max_;
};

} // namespace NPrivate

/**
 * Linearly increasing backoff calculated as:
 * <pre>
 *   next = min(max, min * attempt)
 *   sleep = jitter(next)
 * </pre>
 */
template <typename TJitter>
class TLinearBackoff: private NPrivate::TBackoffBase {
public:
    TLinearBackoff(TDuration min, TDuration max, TJitter jitter = {}) noexcept
        : TBackoffBase(min, max)
        , Jitter_{std::forward<TJitter>(jitter)}
    {
    }

    TDuration operator()() noexcept {
        auto next = Min_ * Attempt_;
        if (next >= Max_) {
            return Jitter_(Max_);
        }

        Attempt_++;
        return Jitter_(next);
    }

    void Reset() noexcept {
        Attempt_ = 1;
    }

    using NPrivate::TBackoffBase::Min;
    using NPrivate::TBackoffBase::Max;

private:
    TJitter Jitter_;
    ui32 Attempt_{1};
};

/**
 * Exponentially increasing backoff calculated as:
 * <pre>
 *   next = min(max, min * 2^attempt)
 *   sleep = jitter(next)
 * </pre>
 */
template <typename TJitter>
class TExpBackoff: private NPrivate::TBackoffBase {
public:
    TExpBackoff(TDuration min, TDuration max, TJitter jitter = {}) noexcept
        : TBackoffBase(min, max)
        , Jitter_{std::forward<TJitter>(jitter)}
        , Current_{min}
    {
    }

    TDuration operator()() noexcept {
        auto next = Current_;
        if (next >= Max_) {
            return Jitter_(Max_);
        }

        Current_ += Current_;
        return Jitter_(next);
    }

    void Reset() noexcept {
        Current_ = Min_;
    }

    using NPrivate::TBackoffBase::Min;
    using NPrivate::TBackoffBase::Max;

private:
    TJitter Jitter_;
    TDuration Current_;
};

} // namespace NSolomon
