#pragma once

#include <util/datetime/base.h>
#include <util/random/fast.h>

namespace NSolomon {

/**
 * Simply returns given value:
 * <pre>
 *   jitter = value
 * </pre>
 */
class TNoJitter {
public:
    TDuration operator()(TDuration value) const noexcept {
        return value;
    }
};

/**
 * Calculates jitter as:
 * <pre>
 *    jitter = value / 2 + random(value / 2)
 * </pre>
 *
 * Not thread-safe.
 * Duration must not be greater than ~ 1 hour.
 */
class THalfJitter {
public:
    THalfJitter() noexcept
        : Rng_{::MicroSeconds()}
    {
    }

    THalfJitter(ui64 seed) noexcept
        : Rng_{seed}
    {
    }

    TDuration operator()(TDuration value) noexcept {
        Y_ASSERT(value < TDuration::MicroSeconds(Max<ui32>()));
        auto half = value / 2;
        auto randomMicros = Rng_.Uniform(static_cast<ui32>(half.MicroSeconds()));
        return half + TDuration::MicroSeconds(randomMicros);
    }

private:
    TFastRng<ui32> Rng_;
};

/**
 * Calculates jitter as:
 * <pre>
 *    random(value)
 * </pre>
 *
 * Not thread-safe.
 * Duration must not be greater than ~ 1 hour.
 */
class TFullJitter {
public:
    TFullJitter() noexcept
        : Rng_{::MicroSeconds()}
    {
    }

    TFullJitter(ui64 seed) noexcept
        : Rng_{seed}
    {
    }

    TDuration operator()(TDuration value) noexcept {
        Y_ASSERT(value < TDuration::MicroSeconds(Max<ui32>()));
        auto randomMicros = Rng_.Uniform(static_cast<ui32>(value.MicroSeconds()));
        return TDuration::MicroSeconds(randomMicros);
    }

private:
    TFastRng<ui32> Rng_;
};

} // namespace NSolomon
