#include "printer.h"

#include <library/cpp/containers/absl_flat_hash/flat_hash_map.h>

#include <util/system/execpath.h>
#include <util/generic/yexception.h>

#include <llvm/DebugInfo/Symbolize/Symbolize.h>
#include <llvm/DebugInfo/Symbolize/DIPrinter.h>
#include <llvm/Support/raw_ostream.h>

namespace NSolomon {
namespace {

class TLlvmBacktracePrinter: public IBacktracePrinter {
public:
    TLlvmBacktracePrinter()
        : BinaryPath{GetPersistentExecPath().begin(),  GetPersistentExecPath().end()}
        , Symbolizer{llvm::symbolize::LLVMSymbolizer::Options()}
    {
    }

    void Print(void* frame, IOutputStream* out) override {
        if (auto it = SymbolsCache.find(frame); it != SymbolsCache.end()) {
            out->Write(it->second);
            return;
        }

        std::string symbolStr;
        {
            llvm::raw_string_ostream stream{symbolStr};
            llvm::symbolize::DIPrinter printer{stream, true, true, false};
            DoPrint(frame, &printer);
        }

        symbolStr.shrink_to_fit();
        SymbolsCache.emplace(frame, symbolStr);
        out->Write(symbolStr);
    }

private:
    void DoPrint(void* frame, llvm::symbolize::DIPrinter* printer) {
        llvm::symbolize::SectionedAddress address;
        address.Address = reinterpret_cast<ui64>(frame) - 1; // last byte of the call instruction

        auto symOrErr = Symbolizer.symbolizeCode(BinaryPath, address);
        Y_ENSURE(symOrErr, "cannot get symbol info for frame" << frame);
        *printer << symOrErr.get();
    }

public:
    const std::string BinaryPath;
    absl::flat_hash_map<void*, std::string> SymbolsCache;
    llvm::symbolize::LLVMSymbolizer Symbolizer;
};

} // namespace

IBacktracePrinter* BacktracePrinter() {
    return Singleton<TLlvmBacktracePrinter>();
}

} // namespace NSolomon
