#pragma once

#include <library/cpp/containers/absl_flat_hash/flat_hash_map.h>

#include <util/datetime/base.h>

namespace NSolomon {

template <
        typename TKey,
        typename TValue,
        typename THashFn = THash<TKey>,
        typename TEqFn = TEqualTo<TKey>>
class TTtlCache {
    struct TEntry: public TIntrusiveListItem<TEntry> {
        TKey Key;
        TValue Value;
        TInstant UpdatedAt;

        TEntry(TKey key, TValue value, TInstant updatedAt) noexcept
            : Key{std::move(key)}
            , Value{std::move(value)}
            , UpdatedAt{updatedAt}
        {
        }
    };

public:
    TTtlCache(TDuration ttl, size_t maxSize) noexcept
        : Ttl_{ttl}
        , MaxSize_{maxSize}
    {
    }

    ~TTtlCache() noexcept {
        while (!Entries_.Empty()) {
            delete Entries_.PopBack();
        }
    }

    std::optional<TValue> Find(const TKey& key, TInstant now) {
        auto it = Index_.find(key);
        if (it == Index_.end()) {
            return std::nullopt;
        }

        TEntry* entry = it->second;
        if ((now - entry->UpdatedAt) < Ttl_) {
            return entry->Value;
        }

        Index_.erase(it);
        std::unique_ptr<TEntry> entryPtr{entry}; // for defer deletion
        entryPtr->Unlink();
        return std::nullopt;
    }

    void Insert(TKey key, TValue value, TInstant now) {
        Y_ENSURE(Entries_.Empty() || now >= Entries_.Front()->UpdatedAt,
                 "invalid now time " << now << ", it must be > " << Entries_.Front()->UpdatedAt);

        if (auto it = Index_.find(key); it != Index_.end()) {
            // update value of the present entry and promote it
            TEntry* prevEntry = it->second;
            prevEntry->Value = std::move(value);
            prevEntry->UpdatedAt = now;

            if (prevEntry != Entries_.Front()) {
                prevEntry->Unlink();
                Entries_.PushFront(prevEntry);
            }
        } else {
            // create a new entry
            auto newEntry = std::make_unique<TEntry>(std::move(key), std::move(value), now);
            Index_.emplace(newEntry->Key, newEntry.get());
            Entries_.PushFront(newEntry.release());
        }

        while (Index_.size() > MaxSize_) {
            // drop oldest entries
            std::unique_ptr<TEntry> entryPtr{Entries_.PopBack()}; // for deref deletion
            Index_.erase(entryPtr->Key);
        }
    }

    size_t Size() const noexcept {
        return Index_.size();
    }

    bool Empty() const noexcept {
        return Index_.empty();
    }

    TDuration Ttl() const noexcept {
        return Ttl_;
    }

    size_t MaxSize() const noexcept {
        return MaxSize_;
    }

private:
    const TDuration Ttl_;
    const size_t MaxSize_;
    absl::flat_hash_map<TKey, TEntry*, THashFn, TEqFn> Index_;
    TIntrusiveList<TEntry> Entries_;
};

} // namespace NSolomon
