#include <library/cpp/testing/gtest/gtest.h>

#include <solomon/libs/cpp/cache/ttl.h>

using namespace NSolomon;

TEST(TTtlCacheTest, Empty) {
    TTtlCache<ui32, ui32> cache{TDuration::Seconds(10), 3};
    ASSERT_TRUE(cache.Empty());
    ASSERT_EQ(cache.Size(), 0u);
    ASSERT_EQ(cache.MaxSize(), 3u);
    ASSERT_EQ(cache.Ttl(), TDuration::Seconds(10));
}

TEST(TTtlCacheTest, Insert) {
    TTtlCache<ui32, ui32> cache{TDuration::Seconds(10), 3};
    auto now = TInstant::Now();

    cache.Insert(1, 11, now);
    ASSERT_EQ(cache.Size(), 1u);

    cache.Insert(2, 22, now);
    ASSERT_EQ(cache.Size(), 2u);

    cache.Insert(3, 33, now);
    ASSERT_EQ(cache.Size(), 3u);

    // overflow
    cache.Insert(4, 44, now);
    ASSERT_EQ(cache.Size(), 3u);
}

TEST(TTtlCacheTest, Find) {
    TTtlCache<ui32, ui32> cache{TDuration::Seconds(10), 3};
    auto now = TInstant::Now();

    cache.Insert(1, 11, now);
    ASSERT_EQ(cache.Size(), 1u);

    // present value
    {
        std::optional<ui32> x = cache.Find(1, now);
        ASSERT_TRUE(x.has_value());
        ASSERT_EQ(*x, 11u);
    }

    // missed value
    {
        std::optional<ui32> x = cache.Find(2, now);
        ASSERT_FALSE(x.has_value());
    }

    cache.Insert(2, 22, now);
    ASSERT_EQ(cache.Size(), 2u);

    // present value
    {
        std::optional<ui32> x = cache.Find(2, now);
        ASSERT_TRUE(x.has_value());
        ASSERT_EQ(*x, 22u);
    }
}

TEST(TTtlCacheTest, Update) {
    TTtlCache<ui32, ui32> cache{TDuration::Seconds(10), 3};
    auto now = TInstant::Now();

    cache.Insert(1, 11, now);
    ASSERT_EQ(cache.Size(), 1u);

    auto x = cache.Find(1, now);
    ASSERT_TRUE(x.has_value());
    ASSERT_EQ(*x, 11u);

    cache.Insert(1, 22, now);
    ASSERT_EQ(cache.Size(), 1u);

    x = cache.Find(1, now);
    ASSERT_TRUE(x.has_value());
    ASSERT_EQ(*x, 22u);
}

TEST(TTtlCacheTest, ExpireOnFind) {
    TTtlCache<ui32, ui32> cache{TDuration::Seconds(10), 3};

    auto t1 = TInstant::Now();
    cache.Insert(1, 11, t1);
    ASSERT_EQ(cache.Size(), 1u);

    auto t2 = t1 + TDuration::Seconds(5);
    cache.Insert(2, 22, t2);
    ASSERT_EQ(cache.Size(), 2u);

    // first item
    {
        // can find first item at {t1 + 5s}
        auto x = cache.Find(1, t1 + TDuration::Seconds(5));
        ASSERT_TRUE(x.has_value());
        ASSERT_EQ(*x, 11u);
        ASSERT_EQ(cache.Size(), 2u);
    }
    {
        // still can find first item at {t1 + 9s}
        auto x = cache.Find(1, t1 + TDuration::Seconds(9));
        ASSERT_TRUE(x.has_value());
        ASSERT_EQ(*x, 11u);
        ASSERT_EQ(cache.Size(), 2u);
    }
    {
        // now first item is expired at {t1 + 10s}
        auto x = cache.Find(1, t1 + TDuration::Seconds(10));
        ASSERT_FALSE(x.has_value());
        ASSERT_EQ(cache.Size(), 1u);
    }

    // second item
    {
        // can find second item at {t2 + 5s}
        auto x = cache.Find(2, t2 + TDuration::Seconds(5));
        ASSERT_TRUE(x.has_value());
        ASSERT_EQ(*x, 22u);
        ASSERT_EQ(cache.Size(), 1u);
    }
    {
        // still can find second item at {t2 + 9s}
        auto x = cache.Find(2, t2 + TDuration::Seconds(9));
        ASSERT_TRUE(x.has_value());
        ASSERT_EQ(*x, 22u);
        ASSERT_EQ(cache.Size(), 1u);
    }
    {
        // now second item is expired at {t2 + 10s}
        auto x = cache.Find(2, t2 + TDuration::Seconds(10));
        ASSERT_FALSE(x.has_value());
        ASSERT_EQ(cache.Size(), 0u);
    }
}

TEST(TTtlCacheTest, UpdateRemovalOrder) {
    TTtlCache<ui32, ui32> cache{TDuration::Seconds(10), 3};

    auto t1 = TInstant::Now();
    cache.Insert(1, 11, t1);
    ASSERT_EQ(cache.Size(), 1u);
    // {1, 11, t1} -> null

    auto t2 = t1 + TDuration::Seconds(1);
    cache.Insert(2, 22, t2);
    ASSERT_EQ(cache.Size(), 2u);
    // {2, 22, t2} -> {1, 11, t1} -> null

    auto t3 = t2 + TDuration::Seconds(1);
    cache.Insert(3, 33, t3);
    ASSERT_EQ(cache.Size(), 3u);
    // {3, 33, t3} -> {2, 22, t2} -> {1, 11, t1} -> null

    // updating '1' will promote it
    auto t4 = t3 + TDuration::Seconds(1);
    cache.Insert(1, 1010, t4);
    ASSERT_EQ(cache.Size(), 3u);
    // {1, 11, t4} -> {3, 33, t3} -> {2, 22, t2} -> null

    // and after adding item '4', the last item '2' was dropped
    auto t5 = t4 + TDuration::Seconds(1);
    cache.Insert(4, 44, t5);
    ASSERT_EQ(cache.Size(), 3u);
    // {4, 44, t5} -> {1, 11, t4} -> {3, 33, t3} -> null

    ASSERT_TRUE(cache.Find(1, t5));
    ASSERT_TRUE(cache.Find(3, t5));
    ASSERT_TRUE(cache.Find(4, t5));

    ASSERT_FALSE(cache.Find(2, t5));
}
