#pragma once

#include <solomon/libs/cpp/ewma/ewma2.h>
#include <solomon/libs/cpp/ewma/ewma_rate.h>

#include <util/generic/string.h>
#include <util/string/builder.h>
#include <util/string/cast.h>

namespace NSolomon {

enum class ECircuitBreakerState: ui8 {
    Closed,
    Open,
    HalfOpen,
};

/**
 * A circuit breaker instance is thread-safe can be used to decorate multiple requests.
 * <p>
 * An instance of {@link ICircuitBreaker} manages the state of a backend system. The circuit breaker
 * is implemented via a finite state machine with three states: Closed, Open and HalfOpen.
 * The circuit breaker does not know anything about the backend's state by itself, but uses the
 * information provided by caller invoking {@link ICircuitBreaker::MarkSuccess} and {@link
 * ICircuitBreaker#MarkFailure} methods. Before communicating with the backend, the permission to do so
 * must be obtained via the method {@link ICircuitBreaker::TryAcquirePermission()}.
 * <p>
 * The state of the circuit breaker changes from Closed to Open when the failure rate is greater than or
 * equal to a (configurable) threshold. Then, all access to the backend is rejected for a (configurable) time
 * duration. No further calls are permitted.
 * <p>
 * After the time duration has elapsed, the circuit breaker state changes from Open to HalfOpen and
 * allows a number of calls to see if the backend is still unavailable or has become available
 * again. If the failure rate is greater than or equal to the configured threshold, the state changes back to Open.
 * If the failure rate is below or equal to the threshold, the state changes back to Closed.
 */
template <typename TDerived, typename TEwma>
class TCircuitBreakerBase {
public:
    /**
     * Construct new circuit breaker which will become opened after percentage of failures exceed {@code maxFailPercent}.
     * Such opened circuit breaker will become half open after {@code resetInterval} has passed.
     *
     * @param maxFailPercent  threshold to open circuit breaker
     * @param resetInterval   try to half close circuit breaker after this interval
     * @param now  current time
     */
    TCircuitBreakerBase(double maxFailPercent, TDuration resetInterval, TInstant now)
        : MaxFailPercent_{maxFailPercent}
        , ResetInterval_{resetInterval}
        , SuccessRate_{resetInterval, now, 0.0}
        , FailRate_{resetInterval, now, 0.0}
    {
        Y_ENSURE(maxFailPercent > 0 && maxFailPercent <= 100, "invalid value of maxFailPercent(" << maxFailPercent << ')');
        Y_ENSURE(resetInterval >= TDuration::Seconds(1), "invalid value of resetInterval(" << resetInterval << ')');
    }

    /**
     * Report about success request.
     *
     * @param now   current time.
     */
    void MarkSuccess(TInstant now) noexcept {
        SuccessRate_.Tick(now);

        auto* self = static_cast<TDerived*>(this);
        switch (self->State()) {
            case ECircuitBreakerState::Closed:
            case ECircuitBreakerState::Open:
                break;

            case ECircuitBreakerState::HalfOpen:
                if (self->SwitchState(now, ECircuitBreakerState::HalfOpen, ECircuitBreakerState::Closed)) {
                    SuccessRate_.Reset(now, 0.0);
                    FailRate_.Reset(now, 0.0);
                }
                break;
        }
    }

    /**
     * Report about failed request and move to {@link EStatus#Open} if failure threshold reached
     *
     * @param now   current time.
     */
    void MarkFailure(TInstant now) noexcept {
        FailRate_.Tick(now);

        auto* self = static_cast<TDerived*>(this);
        switch (self->State()) {
            case ECircuitBreakerState::Closed: {
                double failRate = FailRate_.Rate(now);
                double successRate = SuccessRate_.Rate(now);
                if (double totalRate = failRate + successRate; totalRate != 0.0) {
                    if ((100 * failRate / totalRate) >= MaxFailPercent_) {
                        self->SwitchState(now, ECircuitBreakerState::Closed, ECircuitBreakerState::Open);
                    }
                }
                break;
            }

            case ECircuitBreakerState::HalfOpen:
                self->SwitchState(now, ECircuitBreakerState::HalfOpen, ECircuitBreakerState::Open);
                break;

            case ECircuitBreakerState::Open:
                break;
        }
    }

    /**
     * Acquires a permission to execute a call, only if one is available at the time of invocation.
     *
     * @param now   current time
     * @return {@code true} if a permission was acquired and {@code false} otherwise.
     */
    bool TryAcquirePermission(TInstant now) noexcept {
        auto* self = static_cast<TDerived*>(this);
        switch (self->State()) {
            case ECircuitBreakerState::Closed:
                return true;

            case ECircuitBreakerState::HalfOpen:
                return false;

            case ECircuitBreakerState::Open:
                return self->TryHalfOpen(now, ResetInterval_);
        }

        Y_UNREACHABLE();
    }

    /**
     * Returns human readable summary about circuit breaker state.
     *
     * @param now   current time.
     */
    TString Summary(TInstant now) const noexcept {
        const auto* self = static_cast<const TDerived*>(this);

        const double failRate = FailRate_.Rate(now);
        const double successRate = SuccessRate_.Rate(now);
        const ECircuitBreakerState state = self->State();

        TStringBuilder sb;
        if (double totalRate = failRate + successRate; totalRate != 0.0) {
            double failuresPercent = 100.0 * failRate / totalRate;
            sb << state << " (failRate=" << failRate
                << ", successRate=" << successRate
                << ", failures=" << failuresPercent << "%)";
        } else {
            sb << state << " (unknown failures rate)";
        }
        return sb;
    }

private:
    const double MaxFailPercent_;
    const TDuration ResetInterval_;
    TEwmaRate<TEwma> SuccessRate_;
    TEwmaRate<TEwma> FailRate_;
};

/**
 * Non thread-safe implementation.
 */
class TCircuitBreaker: public TCircuitBreakerBase<TCircuitBreaker, TEwma> {
    friend class TCircuitBreakerBase;
public:
    TCircuitBreaker(double maxFailPercent, TDuration resetInterval, TInstant now) noexcept
        : TCircuitBreakerBase{maxFailPercent, resetInterval, now}
    {
    }

    /**
     * Returns current state of the circuit breaker.
     */
    ECircuitBreakerState State() const noexcept {
        return State_;
    }

    /**
     * Returns point in time when circuit breaker was switched to Open state or TInstant::Zero().
     */
    TInstant OpenedAt() const noexcept {
        return OpenedAt_;
    }

private:
    bool SwitchState(TInstant now, ECircuitBreakerState from, ECircuitBreakerState to) noexcept {
        Y_VERIFY_DEBUG(State_ == from, "state (%s) is not expected (%s)",
            ToString(State_).c_str(),
            ToString(from).c_str());

        if (to == ECircuitBreakerState::Open) {
            OpenedAt_ = now;
        } else if (to == ECircuitBreakerState::Closed) {
            OpenedAt_ = TInstant::Zero();
        }

        State_ = to;
        return true;
    }

    bool TryHalfOpen(TInstant now, TDuration resetInterval) noexcept {
        if ((now - OpenedAt_) >= resetInterval) {
            State_ = ECircuitBreakerState::HalfOpen;
            return true;
        }
        return false;
    }

private:
    ECircuitBreakerState State_{ECircuitBreakerState::Closed};
    TInstant OpenedAt_;
};

/**
 * Thread-safe implementation.
 */
class TAtomicCircuitBreaker: public TCircuitBreakerBase<TAtomicCircuitBreaker, TAtomicEwma> {
    friend class TCircuitBreakerBase;
public:
    TAtomicCircuitBreaker(double maxFailPercent, TDuration resetInterval, TInstant now) noexcept
        : TCircuitBreakerBase{maxFailPercent, resetInterval, now}
    {
    }

    /**
     * Returns current state of the circuit breaker.
     */
    ECircuitBreakerState State() const noexcept {
        return State_.load();
    }

    /**
     * Returns point in time when circuit breaker was switched to Open state or TInstant::Zero().
     */
    TInstant OpenedAt() const noexcept {
        return OpenedAt_.load();
    }

private:
    bool SwitchState(TInstant now, ECircuitBreakerState from, ECircuitBreakerState to) noexcept {
        if (State_.compare_exchange_strong(from, to)) {
            if (to == ECircuitBreakerState::Open) {
                OpenedAt_.store(now);
            } else if (to == ECircuitBreakerState::Closed) {
                OpenedAt_.store(TInstant::Zero());
            }
        }
        return true;
    }

    bool TryHalfOpen(TInstant now, TDuration resetInterval) noexcept {
        if ((now - OpenedAt_.load()) >= resetInterval) {
            State_.store(ECircuitBreakerState::HalfOpen);
            return true;
        }
        return false;
    }

private:
    std::atomic<ECircuitBreakerState> State_{ECircuitBreakerState::Closed};
    std::atomic<TInstant> OpenedAt_;
};

} // namespace NSolomon
