#include <solomon/libs/cpp/circuit_breaker/circuit_breaker.h>

#include <library/cpp/testing/gtest/gtest.h>

#include <util/random/random.h>

using namespace NSolomon;

template <typename T>
class TCircuitBreakerTest: public ::testing::Test {
public:
    TInstant CurrentTime = TInstant::Now();
    T CircuitBreaker{30.0, TDuration::Minutes(1), CurrentTime};
};

using MyTypes = ::testing::Types<TCircuitBreaker, TAtomicCircuitBreaker>;
TYPED_TEST_SUITE(TCircuitBreakerTest, MyTypes);

TYPED_TEST(TCircuitBreakerTest, InitiallyClosed) {
    EXPECT_EQ(this->CircuitBreaker.State(), ECircuitBreakerState::Closed);
}

TYPED_TEST(TCircuitBreakerTest, DoNotOpenOnSuccess) {
    for (int i = 0; i < 100; ++i) {
        this->CircuitBreaker.MarkSuccess(this->CurrentTime);
        EXPECT_EQ(this->CircuitBreaker.State(), ECircuitBreakerState::Closed);
        EXPECT_TRUE(this->CircuitBreaker.TryAcquirePermission(this->CurrentTime));
        this->CurrentTime += TDuration::Seconds(1);
    }
}

TYPED_TEST(TCircuitBreakerTest, OpenOnFailures) {
    // after first failure there is no enough information to change circuit breaker state
    this->CircuitBreaker.MarkFailure(this->CurrentTime);
    EXPECT_EQ(this->CircuitBreaker.State(), ECircuitBreakerState::Closed);
    EXPECT_TRUE(this->CircuitBreaker.TryAcquirePermission(this->CurrentTime));
    this->CurrentTime += TDuration::Seconds(1);

    // after next failure circuit breaker will be opened
    this->CircuitBreaker.MarkFailure(this->CurrentTime);
    EXPECT_EQ(this->CircuitBreaker.State(), ECircuitBreakerState::Open);
    EXPECT_FALSE(this->CircuitBreaker.TryAcquirePermission(this->CurrentTime));
    this->CurrentTime += TDuration::Seconds(1);
}

TYPED_TEST(TCircuitBreakerTest, ReachThreshold) {
    SetRandomSeed(37);

    for (int i = 0; i < 15; ++i) {
        if (RandomNumber<double>() < 0.3) {
            this->CircuitBreaker.MarkFailure(this->CurrentTime);
        } else {
            this->CircuitBreaker.MarkSuccess(this->CurrentTime);
        }
        this->CurrentTime += TDuration::Seconds(1);
    }

    EXPECT_EQ(this->CircuitBreaker.State(), ECircuitBreakerState::Open);
    EXPECT_FALSE(this->CircuitBreaker.TryAcquirePermission(this->CurrentTime));
}

TYPED_TEST(TCircuitBreakerTest, NotReachThreshold) {
    SetRandomSeed(57);

    for (int i = 0; i < 100; ++i) {
        // Only 25% of failures
        if (RandomNumber<double>() < 0.25) {
            this->CircuitBreaker.MarkFailure(this->CurrentTime);
        } else {
            this->CircuitBreaker.MarkSuccess(this->CurrentTime);
        }
        this->CurrentTime += TDuration::Seconds(1);
    }

    EXPECT_TRUE(this->CircuitBreaker.TryAcquirePermission(this->CurrentTime));
    EXPECT_EQ(this->CircuitBreaker.State(), ECircuitBreakerState::Closed);
}

TYPED_TEST(TCircuitBreakerTest, HalfOpenAfterResetInterval) {
    for (int i = 1; i < 10; ++i) {
        this->CircuitBreaker.MarkFailure(this->CurrentTime);
        this->CurrentTime += TDuration::Seconds(1);
    }

    this->CurrentTime += TDuration::Seconds(30);

    EXPECT_EQ(this->CircuitBreaker.State(), ECircuitBreakerState::Open);
    EXPECT_FALSE(this->CircuitBreaker.TryAcquirePermission(this->CurrentTime));
    EXPECT_EQ(this->CircuitBreaker.State(), ECircuitBreakerState::Open);

    this->CurrentTime += TDuration::Seconds(30);

    // state is switched from Open to HalfOpen
    // and only single operation is permitted
    EXPECT_EQ(this->CircuitBreaker.State(), ECircuitBreakerState::Open);
    EXPECT_TRUE(this->CircuitBreaker.TryAcquirePermission(this->CurrentTime));
    EXPECT_EQ(this->CircuitBreaker.State(), ECircuitBreakerState::HalfOpen);

    // next operation is not permitted
    EXPECT_FALSE(this->CircuitBreaker.TryAcquirePermission(this->CurrentTime));
}

TYPED_TEST(TCircuitBreakerTest, OpenAfterFailInHalfOpen) {
    for (int i = 1; i < 10; ++i) {
        this->CircuitBreaker.MarkFailure(this->CurrentTime);
        this->CurrentTime += TDuration::Seconds(1);
    }

    this->CurrentTime += TDuration::Minutes(1);
    EXPECT_TRUE(this->CircuitBreaker.TryAcquirePermission(this->CurrentTime));
    EXPECT_EQ(this->CircuitBreaker.State(), ECircuitBreakerState::HalfOpen);

    this->CircuitBreaker.MarkFailure(this->CurrentTime);
    EXPECT_EQ(this->CircuitBreaker.State(), ECircuitBreakerState::Open);
    EXPECT_FALSE(this->CircuitBreaker.TryAcquirePermission(this->CurrentTime));
    EXPECT_EQ(this->CircuitBreaker.OpenedAt(), this->CurrentTime);
}

TYPED_TEST(TCircuitBreakerTest, CloseAfterSuccessInHalfOpen) {
    for (int i = 1; i < 10; ++i) {
        this->CircuitBreaker.MarkFailure(this->CurrentTime);
        this->CurrentTime += TDuration::Seconds(1);
    }

    this->CurrentTime += TDuration::Minutes(1);
    EXPECT_TRUE(this->CircuitBreaker.TryAcquirePermission(this->CurrentTime));
    EXPECT_EQ(this->CircuitBreaker.State(), ECircuitBreakerState::HalfOpen);

    this->CircuitBreaker.MarkSuccess(this->CurrentTime);
    EXPECT_EQ(this->CircuitBreaker.State(), ECircuitBreakerState::Closed);
    EXPECT_TRUE(this->CircuitBreaker.TryAcquirePermission(this->CurrentTime));
    EXPECT_EQ(this->CircuitBreaker.OpenedAt(), TInstant::Zero());
}
