#include <solomon/libs/cpp/clients/coremon/coremon_client.h>

#include <library/cpp/getopt/last_getopt.h>
#include <library/cpp/monlib/metrics/metric_registry.h>

using namespace NSolomon::NCoremon;

NMonitoring::TMetricRegistry DUMMY_REGISTRY;

class TCommandBase {
public:
    void Execute(int argc, char** argv) {
        NLastGetopt::TOpts opts = Opts();
        NLastGetopt::TOptsParseResult result{&opts, argc, argv};

        yandex::solomon::config::rpc::TGrpcClientConfig conf;
        auto* addr = conf.AddAddresses();
        *addr = TStringBuilder() << Host_ << ':' << Port_;
        DoExecute(
            CreateGrpcClient(std::move(conf), DUMMY_REGISTRY)
        );
    }

    virtual ~TCommandBase() = default;
    virtual NLastGetopt::TOpts Opts() {
        NLastGetopt::TOpts opts;
        opts.AddLongOption('p', "port", "Coremon port to connect to")
            .RequiredArgument("PORT")
            .StoreResult(&Port_)
            .DefaultValue(4710);
        opts.AddLongOption('h', "host", "Coremon host to connect to")
            .RequiredArgument("HOST")
            .StoreResult(&Host_)
            .DefaultValue("localhost");

        return opts;
    }

    virtual TStringBuf Name() const = 0;
    virtual TStringBuf Help() const = 0;

protected:
    virtual void DoExecute(ICoremonClientPtr coremonClient) = 0;

protected:
    TString Host_;
    ui16 Port_;
};

class TShardAssignmentsCommand: public TCommandBase {
    TStringBuf Name() const override {
        return Name_;
    }

    void DoExecute(ICoremonClientPtr coremonClient) override {
        auto assignments = coremonClient->GetShardAssignments().ExtractValueSync();

        if (!assignments.Success()) {
            Cerr << "failed to get shard assignments: " << assignments.Error().Message() << Endl;
            exit(1);
        }

        Cout << "Coremon cluster leader: " << assignments.Value().Leader << "\n\n";

        for (auto&& [host, shards]: assignments.Value().Assignments) {
            Cout << '[' << host.second << "] " << host.first << '\t';
            for (auto shardId: shards) {
                Cout << shardId << ' ';
            }

            Cout << '\n';
        }
    }

    TStringBuf Help() const override {
        return Help_;
    }

private:
    const TString Name_{"dump-assignments"};
    const TString Help_{"Dumps shard assignments received from the coremon's balancer service"};
};

bool IsHelp(TStringBuf arg) {
    return arg == "-h" || arg == "--help" || arg == "help";
}

using TCommands = std::array<THolder<TCommandBase>, 1>;
void PrintUsage(TStringBuf executable, TCommands& commands) {
    Cerr << executable << " <command> [-h|--host] [-p|--port] [OPTIONS]\n\n";

    for (auto& command: commands) {
        Cerr << command->Name() << '\t' << command->Help() << '\n';
    }
}

int main(int argc, char** argv) {
    using namespace NLastGetopt;

    TCommands commands {
        {MakeHolder<TShardAssignmentsCommand>()},
    };

    if (argc <= 1 || IsHelp(argv[1])) {
        PrintUsage(argv[0], commands);
        return 0;
    }

    auto it = FindIf(commands.begin(), commands.end(), [arg = TStringBuf{argv[1]}] (auto&& cmd){
        return cmd->Name() == arg;
    });

    if (it == commands.end()) {
        Cerr << "Unknown command: " << argv[1];
        PrintUsage(argv[0], commands);
        return 1;
    }

    (*it)->Execute(argc, argv);
}
