#include "coremon_client.h"

#include <solomon/libs/cpp/proto_convert/format.h>
#include <solomon/libs/cpp/proto_convert/labels.h>
#include <solomon/protos/coremon/coremon_service.grpc.pb.h>

#include <library/cpp/monlib/metrics/metric_registry.h>

#include <util/datetime/base.h>
#include <util/generic/size_literals.h>
#include <util/string/builder.h>
#include <util/string/cast.h>
#include <util/system/hostname.h>
#include <google/protobuf/text_format.h>

namespace NSolomon::NCoremon {
namespace {
    using namespace grpc;
    using namespace NThreading;
    using namespace NMonitoring;
    using namespace NSolomon;

    using namespace yandex::monitoring::coremon;
    using yandex::solomon::config::rpc::TGrpcClientConfig;

    using NGrpc::TGRpcClientLow;
    using NGrpc::TGRpcClientConfig;
    using NGrpc::TGrpcStatus;
    using NGrpc::TServiceConnection;

    TProcessResult ConvertProcessResponseFromProto(TDataProcessResponse&& resp) {
        TProcessResult native;

        native.Error = std::move(*resp.MutableErrorMessage());
        native.SuccessMetricCount = resp.GetSuccessMetricCount();
        native.Status = resp.GetStatus();

        return native;
    }

    TErrorOr<TShardAssignments, TGenericError> ConvertShardAssignmentsFromProto(TShardAssignmentsResponse&& resp) {
        TShardAssignments result;

        result.Leader = resp.GetLeaderHost();
        if (resp.HostIdsSize() != resp.ShardIdsSize()) {
            return TGenericError{"Unexpected response format: host id and shard id vector sizes don't match"};
        }

        const auto size = resp.HostIdsSize();
        for (auto i = 0u; i < size; ++i) {
            const auto hostId = resp.GetHostIds(i);
            const auto shardId = resp.GetShardIds(i);

            auto host = std::make_pair(resp.GetHosts(hostId), hostId);

            result.Assignments[host].push_back(shardId);
        }

        return result;
    }

    TShardCreated ConvertShardCreatedFromProto(TCreateShardResponse&& resp) {
        TShardCreated result;
        result.ShardId = resp.GetShardId();
        result.Leader = resp.GetLeaderHost();
        result.NumId = resp.GetNumId();
        result.AssignedToHost = resp.GetAssignedToHost();

        return result;
    }

    class TGrpcCoremonClientBase: public ICoremonClient {
    public:
        TGrpcCoremonClientBase(
                IMetricRegistry&,
                std::unique_ptr<TGrpcServiceConnection<CoremonService>> connection)
            : Connection_{std::move(connection)}
        {
        }

        TAsyncDataProcessResponse ProcessPulledData(
            ui32 numId, TString host,
            const NMonitoring::TLabels& optLabels, NMonitoring::EFormat metricFormat,
            TInstant instant, TString response,
            TInstant prevInstant, TString prevResponse) noexcept override
        {
            TPulledDataRequest req;
            req.SetNumId(numId);
            req.SetHost(std::move(host));
            req.SetFormat(ConvertFormatToProto(metricFormat));
            WriteLabelsToProto(*req.MutableOptionalLabels(), optLabels);

            req.SetResponse(std::move(response));
            req.SetResponseTimeMillis(instant.MilliSeconds());
            req.SetPrevResponse(std::move(prevResponse));
            req.SetPrevResponseTimeMillis(prevInstant.MilliSeconds());

            auto promise = NewPromise<TErrorOr<TProcessResult, TApiCallError>>();

            auto cb = [promise] (TGrpcStatus&& status, TDataProcessResponse&& result) mutable {
                if (!status.Ok()) {
                    promise.SetValue(FromGrpcError<TProcessResult>(status));
                } else {
                    promise.SetValue(ConvertProcessResponseFromProto(std::move(result)));
                }
            };

            Connection_->Request<TPulledDataRequest, TDataProcessResponse>(
                req,
                std::move(cb),
                &CoremonService::Stub::AsyncProcessPulledData);

            return promise.GetFuture();
        }

        TAsyncDataProcessResponse ProcessPushedData(
                ui32 numId,
                NMonitoring::EFormat format,
                TInstant ts,
                TString content) noexcept override
        {
            TPushedDataRequest req;
            req.SetNumId(numId);
            req.SetFormat(ConvertFormatToProto(format));
            req.SetContent(std::move(content));
            req.SetTimeMillis(ts.MilliSeconds());

            auto promise = NewPromise<TErrorOr<TProcessResult, TApiCallError>>();

            auto cb = [promise] (TGrpcStatus&& status, TDataProcessResponse&& result) mutable {
                if (!status.Ok()) {
                    promise.SetValue(FromGrpcError<TProcessResult>(status));
                } else {
                    promise.SetValue(ConvertProcessResponseFromProto(std::move(result)));
                }
            };

            Connection_->Request<TPushedDataRequest, TDataProcessResponse>(
                req,
                std::move(cb),
                &CoremonService::Stub::AsyncProcessPushedData);

            return promise.GetFuture();
        }

        TAsyncShardAssignments GetShardAssignments() noexcept override {
            TShardAssignmentsRequest req;
            req.SetTtl(2);

            auto promise = NewPromise<TErrorOr<TShardAssignments, TApiCallError>>();

            auto cb = [promise] (TGrpcStatus&& status, TShardAssignmentsResponse&& resp) mutable {
                if (!status.Ok()) {
                    promise.SetValue(FromGrpcError<TShardAssignments>(status));
                } else {
                    auto result = ConvertShardAssignmentsFromProto(std::move(resp));
                    if (result.Success()) {
                        promise.SetValue(result.Extract());
                    } else {
                        promise.SetValue({result.ExtractError()});
                    }
                }
            };

            Connection_->Request<TShardAssignmentsRequest, TShardAssignmentsResponse>(
                std::move(req),
                std::move(cb),
                &CoremonService::Stub::AsyncGetShardAssignments);

            return promise;
        };

        TAsyncShardCreated CreateShard(
                TString projectId,
                TString serviceName,
                TString clusterName,
                TString createdBy) noexcept override
        {
            TCreateShardRequest req;
            req.SetTtl(2);
            req.SetProjectId(std::move(projectId));
            req.SetServiceName(std::move(serviceName));
            req.SetClusterName(std::move(clusterName));
            req.SetCreatedBy(std::move(createdBy));

            auto promise = NewPromise<TErrorOr<TShardCreated, TApiCallError>>();

            auto cb = [promise] (TGrpcStatus&& status, TCreateShardResponse&& resp) mutable {
                if (!status.Ok()) {
                    promise.SetValue(FromGrpcError<TShardCreated>(status));
                } else {
                    promise.SetValue(ConvertShardCreatedFromProto(std::move(resp)));
                }
            };

            Connection_->Request<TCreateShardRequest, TCreateShardResponse>(
                std::move(req),
                std::move(cb),
                &CoremonService::Stub::AsyncCreateShard);

            return promise;
        }

        void Stop(bool) noexcept override {
        }

    private:
        std::unique_ptr<TGrpcServiceConnection<CoremonService>> Connection_;
    };

    ICoremonClientPtr CreateCoremonClient(
            IMetricRegistry& registry,
            std::unique_ptr<TGrpcServiceConnection<CoremonService>> connection)
    {
        return std::make_unique<TGrpcCoremonClientBase>(registry, std::move(connection));
    }
}

    ICoremonClientPtr CreateGrpcClient(const TGrpcClientConfig& conf, TMetricRegistry& registry, TString clientId) {
        auto threadPool = CreateGrpcThreadPool(conf);
        auto sc = CreateGrpcServiceConnection<CoremonService>(
            conf,
            false,
            registry,
            std::move(threadPool),
            std::move(clientId));

        return CreateCoremonClient(registry, std::move(sc));
    }

    ICoremonClusterClientPtr CreateGrpcClusterClient(const TGrpcClientConfig& conf, TMetricRegistry& registry, TString clientId) {
        return std::make_shared<TGrpcClusterClientBase<CoremonService, ICoremonClient>>(
            conf,
            false,
            registry,
            CreateCoremonClient,
            std::move(clientId));
    }

} // namespace NSolomon::NCoremon
