#include <solomon/libs/cpp/clients/ingestor/ingestor_client.h>
#include <solomon/libs/cpp/host_resolver/host_resolver.h>
#include <solomon/libs/cpp/proto_convert/labels.h>
#include <solomon/libs/cpp/proto_convert/format.h>

#include <library/cpp/colorizer/colors.h>
#include <library/cpp/monlib/metrics/metric_registry.h>
#include <library/cpp/getopt/last_getopt.h>
#include <library/cpp/getopt/modchooser.h>

#include <util/stream/file.h>
#include <util/folder/path.h>

using namespace NLastGetopt;
using namespace NSolomon;
using namespace NIngestor;

using NMonitoring::EFormat;

NMonitoring::TMetricRegistry DUMMY_REGISTRY;

class TCommandBase: public TMainClassArgs {
protected:
    virtual void DoRegisterOptions(NLastGetopt::TOpts& opts) = 0;
    virtual int DoExecute(
            NLastGetopt::TOptsParseResult&& opts,
            IIngestorClientPtr ingestorClient) = 0;


    int DoRun(NLastGetopt::TOptsParseResult&& opts) override {
        yandex::solomon::config::rpc::TGrpcClientConfig conf;

        auto* addr = conf.AddAddresses();
        *addr = TStringBuilder() << Host_ << ':' << Port_;

        return DoExecute(
            std::move(opts),
            CreateIngestorGrpcClient(conf, DUMMY_REGISTRY)
        );
    }

private:
    void RegisterOptions(NLastGetopt::TOpts& opts) override {
        opts.AddLongOption('p', "port", "Ingestor port to connect to")
                .RequiredArgument("PORT")
                .StoreResult(&Port_)
                .DefaultValue(4760);
        opts.AddLongOption('h', "host", "Ingestor host to connect to")
                .RequiredArgument("HOST")
                .StoreResult(&Host_)
                .DefaultValue("localhost");

        DoRegisterOptions(opts);
    }

protected:
    TString Host_;
    ui16 Port_;
};

class TListShards: public TCommandBase {
protected:
    void DoRegisterOptions(NLastGetopt::TOpts& opts) override {
        opts.SetTitle("list currently know shard ids");
    }

    int DoRun(NLastGetopt::TOptsParseResult&& opts) override {
        Y_UNUSED(opts);
        yandex::monitoring::ingestor::TGetAssignedShardsRequest req;
        auto addrs = ResolveAddress();
        if (addrs.empty()) {
            Cerr << NColorizer::RED << "No hosts in this conductor group" << NColorizer::RESET << Endl;
            return 1;
        }

        yandex::solomon::config::rpc::TGrpcClientConfig conf;
        for (const auto& addr : addrs) {
            *(conf.AddAddresses()) = addr;
        }
        auto client = CreateIngestorGrpcClusterClient(conf, DUMMY_REGISTRY);

        int err = 0;
        for (auto& addr : addrs) {
            auto res = client->Get(addr)->GetAssignedShards(req).ExtractValueSync();

            if (res->Success()) {
                auto& val = res->Value();
                for (int i = 0; i < val.numids_size(); i++) {
                    Cout << addr << '\t' << val.numids(i) << Endl;
                }
            } else {
                Cerr << NColorizer::RED << "Error code " << res->Error().GRpcStatusCode << ": " << NColorizer::RESET;
                Cerr << res->Error().Msg << Endl;
                err = 1;
            }
        }
        return err;
    }

private:
    int DoExecute(
        NLastGetopt::TOptsParseResult&& opts,
        IIngestorClientPtr ingestorClient) override {
        Y_UNUSED(opts);
        Y_UNUSED(ingestorClient);
        return 0;
    }

    TAddressSet ResolveAddress() {
        TAddressSet res;
        ConductorGroupResolver()->Resolve(Host_, &res);
        return res;
    }
};

class TProcessPulledDataCommand: public TCommandBase {
private:
    void DoRegisterOptions(NLastGetopt::TOpts& opts) override {
        opts.AddLongOption("num-id", "NumId value")
                .RequiredArgument("NUMBER");
        opts.AddLongOption("source-host")
                .RequiredArgument("HOST");
        opts.AddLongOption("source-id")
                .RequiredArgument("NUMBER");
        opts.AddLongOption("optional-labels")
                .OptionalArgument("KEY=VALUE");
        opts.AddLongOption("pulled-ts")
                .OptionalArgument("TIMESTAMP");
        opts.AddLongOption("json-data", "json file with metrics")
                .OptionalArgument("FILE");
        opts.AddLongOption("spack-data", "json file with metrics")
                .OptionalArgument("FILE");
    }

    std::pair<TString, EFormat> GetDataAndFormat(NLastGetopt::TOptsParseResult& opts) const {
        TStringBuf filePath;
        EFormat format;

        if (opts.Has("json-data")) {
            filePath = opts.Get("json-data");
            format = EFormat::JSON;
        } else if (opts.Has("spack-data")) {
            filePath = opts.Get("spack-data");
            format = EFormat::SPACK;
        } else {
            ythrow yexception() << "either json-data or spack-data must be specified";
        }

        Y_ENSURE(!filePath.empty(), "file path cannot be empty");

        TFileInput file{TString{filePath}};
        return {file.ReadAll(), format};
    }

    int DoExecute(
            NLastGetopt::TOptsParseResult&& opts,
            IIngestorClientPtr ingestorClient) override
    {
        auto numId = opts.Get<ui32>("num-id");
        auto host = opts.Get<TString>("host");
        auto sourceId = opts.Get<ui64>("source-id");

        NMonitoring::TLabels optLabels;
        if (opts.Has("optional-labels")) {
            auto optLabelsStr = opts.Get<TString>("optional-labels");

            if (!optLabelsStr.empty()) {
                for (auto&& it: StringSplitter(optLabelsStr).Split(',').SkipEmpty()) {
                    TVector<TString> keyValue = StringSplitter(it.Token()).Split('=').Limit(2);
                    Y_ENSURE(keyValue.size() == 2, "wrong label format: " << it.Token());

                    optLabels.Add(keyValue[0], keyValue[1]);
                }
            }
        }

        TInstant pulledTs;
        if (opts.Has("pulled-ts")) {
            auto pulledTsSeconds = opts.Get<ui64>("pulled-ts");
            pulledTs = TInstant::Seconds(pulledTsSeconds);
        } else {
            pulledTs = TInstant::Now();
        }

        auto [data, format] = GetDataAndFormat(opts);

        yandex::monitoring::ingestor::TPulledDataRequest req;
        req.SetNumId(numId);
        req.AddHost(std::move(host));
        req.AddFormat(ConvertFormatToProto(format));
        req.AddSourceId(sourceId);
        WriteLabelsToProto(*req.AddOptionalLabels(), optLabels);
        req.AddResponse(std::move(data));
        req.AddResponseTimeMillis(pulledTs.MilliSeconds());

        auto f = ingestorClient->ProcessPulledData(req);

        auto res = f.ExtractValueSync();
        if (!res.Success()) {
            Cerr << "failed to process pulled data: " << res.Error().Message() << Endl;
            return 1;
        } else {
            auto val = res.Value();

            Cout << "SuccessMetricCount: " << val.GetSuccessMetricCount(0) << Endl;
            Cout << "Status: " << UrlStatusType_Name(val.GetStatus(0)) << Endl;
            Cout << "Error: " << val.GetErrorMessage(0) << Endl;

            return 0;
        }
    }
};

class TProcessPushedDataCommand: public TCommandBase {
private:
    void DoRegisterOptions(NLastGetopt::TOpts& opts) override {
        opts.AddLongOption("num-id", "NumId value")
                .RequiredArgument("NUMBER");
        opts.AddLongOption("json-data", "json file with metrics")
                .OptionalArgument("FILE");
        opts.AddLongOption("spack-data", "json file with metrics")
                .OptionalArgument("FILE");
        opts.AddLongOption("pushed-ts")
                .OptionalArgument("TIMESTAMP");
    }

    std::pair<TString, EFormat> GetDataAndFormat(NLastGetopt::TOptsParseResult& opts) const {
        TStringBuf filePath;
        EFormat format;

        if (opts.Has("json-data")) {
            filePath = opts.Get("json-data");
            format = EFormat::JSON;
        } else if (opts.Has("spack-data")) {
            filePath = opts.Get("spack-data");
            format = EFormat::SPACK;
        } else {
            ythrow yexception() << "either json-data or spack-data must be specified";
        }

        Y_ENSURE(!filePath.empty(), "file path cannot be empty");

        TFileInput file{TString{filePath}};
        return {file.ReadAll(), format};
    }

    int DoExecute(
            NLastGetopt::TOptsParseResult&& opts,
            IIngestorClientPtr ingestorClient) override
    {
        auto numId = opts.Get<ui32>("num-id");
        auto [data, format] = GetDataAndFormat(opts);

        TInstant pushedTs;
        if (opts.Has("pushed-ts")) {
            auto pushedTsSeconds = opts.Get<ui64>("pushed-ts");
            pushedTs = TInstant::Seconds(pushedTsSeconds);
        } else {
            pushedTs = TInstant::Now();
        }

        yandex::monitoring::ingestor::TPushedDataRequest req;
        req.SetNumId(numId);
        req.SetFormat(ConvertFormatToProto(format));
        req.SetContent(std::move(data));
        req.SetTimeMillis(pushedTs.MilliSeconds());

        auto f = ingestorClient->ProcessPushedData(req);

        auto res = f.ExtractValueSync();
        if (!res.Success()) {
            Cerr << "failed to process pushed data: " << res.Error().Message() << Endl;
            return 1;
        } else {
            auto val = res.Value();

            Cout << "SuccessMetricCount: " << val.GetSuccessMetricCount(0) << Endl;
            Cout << "Status: " << UrlStatusType_Name(val.GetStatus(0)) << Endl;
            Cout << "Error: " << val.GetErrorMessage(0) << Endl;

            return 0;
        }
    }
};


template <class TCommand>
TMainClassArgs* GetCommand() {
    return Singleton<TCommand>();
}

int Main(int argc, const char** argv) {
    NLastGetopt::NComp::TCustomCompleter::FireCustomCompleter(argc, argv);

    TModChooser modeChooser;
    modeChooser.SetDescription("ingestor_cli -- a console client for the Ingestor gRPC service");
    modeChooser.SetPrintShortCommandInUsage(false);
    modeChooser.SetModesHelpOption("-h");

    modeChooser.AddGroupModeDescription("observe");

    modeChooser.AddMode("list-shards", new TListShards{}, "list currently know shards");
    modeChooser.AddAlias("ls", "list-shards");

    modeChooser.AddGroupModeDescription("gRPC commands", false);
    modeChooser.AddMode(
            "process-pulled-data",
            GetCommand<TProcessPulledDataCommand>(),
            "Sends data to the ProcessPulledData endpoint",
            false,
            false);
    modeChooser.AddMode(
            "process-pushed-data",
            GetCommand<TProcessPushedDataCommand>(),
            "Sends data to the ProcessPushedData endpoint",
            false,
            false);

    modeChooser.AddGroupModeDescription("internal operations", false);
    modeChooser.AddCompletions("ingestor_cli", "completion", false, true);

    return modeChooser.Run(argc, argv);
};

int main(int argc, const char** argv) {
    try {
        return Main(argc, argv);
    } catch (...) {
        Cerr << "Unhandled exception: " << CurrentExceptionMessage() << ". Process will terminate" << Endl;
        return 1;
    }
}
